<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

require __DIR__ . '/vendor/autoload.php';
$config     = (include './config/database.php');
$app        = require_once __DIR__ . '/bootstrap/app.php';
$stack      = GuzzleHttp\HandlerStack::create();
$httpClient = new GuzzleHttp\Client([
    'handler' => $stack,
]);

function url()
{
    if (isset($_SERVER['HTTPS']))
    {
        $protocol = ($_SERVER['HTTPS'] && $_SERVER['HTTPS'] != "off") ? "https" : "http";
        return $protocol . "://" . $_SERVER['HTTP_HOST'];
    }
    else
    {
        $protocol = 'http';
        return $protocol . "://" . $_SERVER['HTTP_HOST'] . str_replace('/ficheck.php', '', $_SERVER['PHP_SELF']);
    }

}

// Get the protocol
$protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https://" : "http://";

// Get the domain name
$domain = $_SERVER['HTTP_HOST'];

// Get the current directory
$directory = dirname($_SERVER['PHP_SELF']);

// Concatenate all parts to form the full URL
$url = $protocol . $domain . $directory;

// Check if the last character is a '/'
if (substr($url, -1) === '/')
{
    // Remove the last character
    $url = substr($url, 0, -1);
}

try
{
    $apiResponse = $httpClient->get($url . '/api/v1/fi-check/addons');
    if ($apiResponse->getStatusCode() == 200)
    {
        $addons = json_decode($apiResponse->getBody());
    }
    else
    {
        $addons = [];
    }
}
catch (Exception $e)
{
    $addons = [];
}

try
{
    $apiResponse = $httpClient->get($url . '/api/v1/fi-check/version');
    if ($apiResponse->getStatusCode() == 200)
    {
        $version = json_decode($apiResponse->getBody());
    }
    else
    {
        $version = [];
    }
}
catch (Exception $e)
{
    $version = [];
}

try
{
    $apiResponse = $httpClient->get($url . '/api/v1/fi-check/migration');

    if ($apiResponse->getStatusCode() == 200)
    {
        $migration = json_decode($apiResponse->getBody());
    }
    else
    {
        $migration = [];
    }
}
catch (Exception $e)
{
    $migration = [];
}

ob_start();
phpinfo(INFO_MODULES);
$info = ob_get_contents();
ob_end_clean();
$info = stristr($info, 'Client API version');
preg_match('/[1-9].[0-9].[1-9][0-9]/', $info, $match);
$mySqlVersion = $match[0];

function logReader($filename, $lines = 600, $buffer = 4096)
{
    // Open the file
    $f = fopen($filename, "rb");

    // Jump to last character
    fseek($f, -1, SEEK_END);

    // Read it and adjust line number if necessary
    // (Otherwise the result would be wrong if file doesn't end with a blank line)
    if (fread($f, 1) != "\n") $lines -= 1;

    // Start reading
    $output = '';
    $chunk  = '';

    // While we would like more
    while (ftell($f) > 0 && $lines >= 0)
    {
        // Figure out how far back we should jump
        $seek = min(ftell($f), $buffer);

        // Do the jump (backwards, relative to where we are)
        fseek($f, -$seek, SEEK_CUR);

        // Read a chunk and prepend it to our output
        $output = ($chunk = fread($f, $seek)) . $output;

        // Jump back to where we started reading
        fseek($f, -mb_strlen($chunk, '8bit'), SEEK_CUR);

        // Decrease our line counter
        $lines -= substr_count($chunk, "\n");
    }

    // While we have too many lines
    // (Because of buffer size we might have read too many)
    while ($lines++ < 0)
    {
        // Find first newline and remove all text before that
        $output = substr($output, strpos($output, "\n") + 1);
    }

    // Close file and return
    fclose($f);
    return $output;
}

// -----------------------------------------------------------------------------
// Setup
// -----------------------------------------------------------------------------
set_time_limit(120); // 2 minutes

$options = [];

// Show or hide the server name and IP address
$showServerName = false;

// Optional: mysql performance test
$default_database = $config['default'];
$database_config  = $config['connections'][$default_database];
$options          = [
    'db.host' => $database_config['host'],
    'db.user' => $database_config['username'],
    'db.pw'   => $database_config['password'],
    'db.name' => $database_config['database'],
];

// -----------------------------------------------------------------------------
// Main
// -----------------------------------------------------------------------------
// check performance
$benchmarkResult = test_benchmark($options);

// benchmark.php?json
if (isset($_GET['json']))
{
    // Json output
    header('Content-Type', 'application/json');
    echo json_encode($benchmarkResult, JSON_PRETTY_PRINT);
}
else
{
    // html output
    //    echo print_html_result($benchmarkResult, $showServerName);
}

// -----------------------------------------------------------------------------
// Benchmark functions
// -----------------------------------------------------------------------------

function test_benchmark(array $settings)
{
    $result                           = [];
    $result['version']                = '1.6';
    $result['sysinfo']['time']        = date('Y-m-d H:i:s');
    $result['sysinfo']['php_version'] = PHP_VERSION;
    $result['sysinfo']['platform']    = PHP_OS;
    $result['sysinfo']['server_name'] = isset($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : 'N/A';
    $result['sysinfo']['server_addr'] = isset($_SERVER['SERVER_ADDR']) ? $_SERVER['SERVER_ADDR'] : 'N/A';
    $result['sysinfo']['xdebug']      = in_array('xdebug', get_loaded_extensions());

    $timeStart = microtime(true);

    test_math($result);
    test_string($result);
    test_loops($result);
    test_ifelse($result);

    $result['benchmark']['calculation'] = timer_diff($timeStart) . ' sec.';

    if (isset($settings['db.host']))
    {
        test_mysql($result, $settings);
    }

    $result['benchmark']['total'] = timer_diff($timeStart) . ' sec.';

    return $result;
}

function test_math(&$result, $count = 99999)
{
    $timeStart = microtime(true);

    for ($i = 0; $i < $count; $i++)
    {
        sin($i);
        asin($i);
        cos($i);
        acos($i);
        tan($i);
        atan($i);
        abs($i);
        floor($i);
        exp($i);
        is_finite($i);
        is_nan($i);
        sqrt($i);
        log10($i);
    }

    $result['benchmark']['math'] = timer_diff($timeStart) . ' sec.';
}

function test_string(&$result, $count = 99999)
{
    $timeStart = microtime(true);

    $string = 'the quick brown fox jumps over the lazy dog';
    for ($i = 0; $i < $count; $i++)
    {
        addslashes($string);
        chunk_split($string);
        metaphone($string);
        strip_tags($string);
        md5($string);
        sha1($string);
        strtoupper($string);
        strtolower($string);
        strrev($string);
        strlen($string);
        soundex($string);
        ord($string);
    }
    $result['benchmark']['string'] = timer_diff($timeStart) . ' sec.';
}

function test_loops(&$result, $count = 999999)
{
    $timeStart = microtime(true);
    for ($i = 0; $i < $count; ++$i)
    {
    }

    $i = 0;
    while ($i < $count)
    {
        ++$i;
    }

    $result['benchmark']['loops'] = timer_diff($timeStart) . ' sec.';
}

function test_ifelse(&$result, $count = 999999)
{
    $timeStart = microtime(true);
    for ($i = 0; $i < $count; $i++)
    {
        if ($i == -1)
        {
        }
        elseif ($i == -2)
        {
        }
        else
        {
            if ($i == -3)
            {
            }
        }
    }
    $result['benchmark']['ifelse'] = timer_diff($timeStart) . ' sec.';
}

function test_mysql(&$result, $settings)
{
    $timeStart = microtime(true);

    $link                                 = mysqli_connect($settings['db.host'], $settings['db.user'], $settings['db.pw']);
    $result['benchmark']['mysql_connect'] = timer_diff($timeStart) . ' sec.';

    mysqli_select_db($link, $settings['db.name']);
    $result['benchmark']['mysql_select_db'] = timer_diff($timeStart) . ' sec.';

    $dbResult                                   = mysqli_query($link, 'SELECT VERSION() as version;');
    $arr_row                                    = mysqli_fetch_array($dbResult);
    $result['sysinfo']['mysql_version']         = $arr_row['version'];
    $result['benchmark']['mysql_query_version'] = timer_diff($timeStart) . ' sec.';

    $query = "SELECT BENCHMARK(1000000, AES_ENCRYPT('hello', UNHEX('F3229A0B371ED2D9441B830D21A390C3')));";
    mysqli_query($link, $query);
    $result['benchmark']['mysql_query_benchmark'] = timer_diff($timeStart) . ' sec.';

    mysqli_close($link);

    $result['benchmark']['mysql_total'] = timer_diff($timeStart) . ' sec.';

    return $result;
}

function timer_diff($timeStart)
{
    return number_format(microtime(true) - $timeStart, 3);
}

function h($v)
{
    return htmlentities($v);
}

function getFreeSpace()
{
    $diskFreeSpace = disk_free_space(base_path());
    return round(($diskFreeSpace / 1000000), 2);
}


$requirements1 = [
    [
        'requirement' => 'Free Space',
        'required'    => '100 MB',
        'actual'      => (getFreeSpace() > 100 ? getFreeSpace() . ' MB' : 'No'),
        'result'      => (getFreeSpace() > 100 ? 1 : 0),
    ],
    [
        'requirement' => 'PHP Version',
        'required'    => '8.1.0',
        'actual'      => PHP_VERSION,
        'result'      => ((version_compare(PHP_VERSION, '8.1.0') >= 0) ? 1 : 0),
    ],
    [
        'requirement' => 'Fileinfo Extension',
        'required'    => 'Yes',
        'actual'      => ((extension_loaded('fileinfo')) ? 'Yes' : 'No'),
        'result'      => ((extension_loaded('fileinfo')) ? 1 : 0),
    ],
    [
        'requirement' => 'OpenSSL Extension',
        'required'    => 'Yes',
        'actual'      => ((extension_loaded('openssl')) ? 'Yes' : 'No'),
        'result'      => ((extension_loaded('openssl')) ? 1 : 0),
    ],
    [
        'requirement' => 'PDO Extension',
        'required'    => 'Yes',
        'actual'      => ((extension_loaded('pdo')) ? 'Yes' : 'No'),
        'result'      => ((extension_loaded('pdo')) ? 1 : 0),
    ],
    [
        'requirement' => 'PDO MySQL Extension',
        'required'    => 'Yes',
        'actual'      => ((extension_loaded('pdo_mysql')) ? 'Yes' : 'No'),
        'result'      => ((extension_loaded('pdo_mysql')) ? 1 : 0),
    ],
    [
        'requirement' => 'MBString Extension',
        'required'    => 'Yes',
        'actual'      => ((extension_loaded('mbstring')) ? 'Yes' : 'No'),
        'result'      => ((extension_loaded('mbstring')) ? 1 : 0),
    ],
    [
        'requirement' => 'Tokenizer Extension',
        'required'    => 'Yes',
        'actual'      => ((extension_loaded('tokenizer')) ? 'Yes' : 'No'),
        'result'      => ((extension_loaded('tokenizer')) ? 1 : 0),
    ],
    [
        'requirement' => 'Graphics Drawing Extension',
        'required'    => 'Yes',
        'actual'      => ((extension_loaded('gd')) ? 'Yes' : 'No'),
        'result'      => ((extension_loaded('gd')) ? 1 : 0),
    ],
    [
        'requirement' => 'XML PHP Extension',
        'required'    => 'Yes',
        'actual'      => ((extension_loaded('xml')) ? 'Yes' : 'No'),
        'result'      => ((extension_loaded('xml')) ? 1 : 0),
    ],
    [
        'requirement' => 'DOM PHP Extension',
        'required'    => 'Yes',
        'actual'      => ((extension_loaded('dom')) ? 'Yes' : 'No'),
        'result'      => ((extension_loaded('dom')) ? 1 : 0),
    ],
    [
        'requirement' => 'Iconv PHP Extension',
        'required'    => 'Yes',
        'actual'      => ((extension_loaded('iconv')) ? 'Yes' : 'No'),
        'result'      => ((extension_loaded('iconv')) ? 1 : 0),
    ],
    [
        'requirement' => 'cURL PHP Extension',
        'required'    => 'Yes',
        'actual'      => ((extension_loaded('curl')) ? 'Yes' : 'No'),
        'result'      => ((extension_loaded('curl')) ? 1 : 0),
    ],
];

// Mod Rewrite only supported by Apache-based webservers.
if (function_exists('apache_get_modules'))
{
    $modRewriteRequirement = [
        [
            'requirement' => 'Mod Rewrite Extension for SEO friendly URL',
            'required'    => 'Yes',
            'actual'      => ((in_array('mod_rewrite', apache_get_modules())) ? 'Yes' : 'No'),
            'result'      => ((in_array('mod_rewrite', apache_get_modules())) ? 1 : 0),
        ],
    ];
}
else
{
    $modRewriteRequirement = [
        [
            'requirement' => 'Mod Rewrite Extension for SEO friendly URL',
            'required'    => 'No',
            'actual'      => 'Unable to access method apache_get_modules()',
            'result'      => '0',
        ],
    ];
}

$requirements2 = [
    [
        'requirement' => 'Webserver',
        'required'    => 'Yes',
        'actual'      => $_SERVER["SERVER_SOFTWARE"],
        'result'      => 1,
    ],
    [
        'requirement' => 'Database Version',
        'required'    => 'Yes',
        'actual'      => 'MySql : ' . h($benchmarkResult['sysinfo']['mysql_version']),
        'result'      => 1,
    ],
];

$requirements3 = [
    [
        'requirement' => 'Version Number',
        'required'    => '-',
        'actual'      => isset($version->data->version) ? $version->data->version : '',
        'result'      => 1,
    ],
    [
        'requirement' => 'Key',
        'required'    => '-',
        'actual'      => isset($version->data->key) ? $version->data->key : '',
        'result'      => 1,
    ],
    [
        'requirement' => 'Last Migration',
        'required'    => '-',
        'actual'      => isset($migration->data->migration) ? $migration->data->migration : '',
        'result'      => 1,
    ],
];

$requirements = array_merge($requirements1, $modRewriteRequirement, $requirements2, $requirements3);

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <title>FusionInvoice Check</title>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <meta name="robots" content="noindex, follow">

    <link rel="apple-touch-icon" sizes="180x180" href="apple-touch-icon.png">
    <link rel="icon" type="image/png" sizes="32x32" href="favicon-32x32.png">
    <link rel="icon" type="image/png" sizes="16x16" href="favicon-16x16.png">
    <link rel="manifest" href="site.webmanifest">
    <link rel="mask-icon" href="safari-pinned-tab.svg" color="#5bbad5">
    <meta name="msapplication-TileColor" content="#da532c">
    <meta name="theme-color" content="#ffffff">

    <!-- Google Font: Source Sans Pro -->
    <link rel="stylesheet" href="assets/dist/css/fonts.google.css">
    <!-- Font Awesome -->
    <link rel="stylesheet" href="assets/plugins/fontawesome-free/css/all.min.css">
    <!-- Tempusdominus Bootstrap 4 -->
    <link rel="stylesheet"
          href="assets/plugins/tempusdominus-bootstrap-4/css/tempusdominus-bootstrap-4.min.css">
    <!-- iCheck -->
    <link rel="stylesheet" href="assets/plugins/icheck-bootstrap/icheck-bootstrap.min.css">
    <!-- Theme style -->
    <link rel="stylesheet" href="assets/dist/css/adminlte.min.css">
    <!-- overlayScrollbars -->
    <link rel="stylesheet" href="assets/plugins/overlayScrollbars/css/OverlayScrollbars.min.css">
</head>
<body>
<div class="wrapper">
    <section class="content-header">
        <div class="container-fluid">
            <div class="row mb-2">
                <div class="col-sm-12">
                    <h1 class="text-center">FI Config Check</h1>
                </div>
            </div>
        </div>
    </section>
    <section class="content">
        <div class="container-fluid">
            <div class="row mb-2">
                <div class="col-12">
                    <div class="card">
                        <div class="card-header">
                            <h6>Welcome to FusionInvoice
                                <?php echo isset($version->data->version) ? $version->data->version : '--'; ?>
                                System Check.
                            </h6>
                        </div>
                        <div class="card-body p-0">
                            <div class="row">
                                <div class="col-12  table-responsive">
                                    <table class="table table-sm table-striped">
                                        <tr>
                                            <th>Requirement</th>
                                            <th>Required</th>
                                            <th>Actual</th>
                                            <th>Result</th>
                                        </tr>
                                        <?php foreach ($requirements as $requirement)
                                        { ?>
                                            <tr>
                                                <td><?php echo $requirement['requirement']; ?></td>
                                                <td><?php echo $requirement['required']; ?></td>
                                                <td><?php echo $requirement['actual']; ?></td>
                                                <td><?php if ($requirement['result'] == 1)
                                                    { ?><span style="color: green;">Pass</span><?php }
                                                    else
                                                    { ?><span style="color: red;">Fail</span><?php } ?></td>
                                            </tr>
                                        <?php } ?>
                                        <?php
                                        if (isset($addons->data) && count($addons->data) > 0)
                                        {
                                            foreach ($addons->data as $key => $addon)
                                            {
                                                ?>
                                                <tr>
                                                    <td><strong><?php echo $key == 0 ? 'Add-ons' : ''; ?></strong></td>
                                                    <td>-</td>
                                                    <td><?php echo $addon->name; ?></td>
                                                    <td><span style="color: green;">Pass</span></td>
                                                </tr>
                                                <?php
                                            }
                                        }
                                        ?>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="card">
                        <div class="card-body p-0">
                            <div class="row  py-3">
                                <div class="col-12 text-center text-bold">
                                    <h4 class="mb-0">phpMySqlBenchmark</h4>
                                </div>
                            </div>
                            <div class="row">
                                <div class="col-md-12">
                                    <table class="table table-sm table-striped">
                                        <thead>
                                        <tr>
                                            <th width="50%">System Info</th>
                                            <th width="50%"></th>
                                        </tr>
                                        </thead>
                                        <tbody>
                                        <tr class="even">
                                            <td>Version</td>
                                            <td> <?php echo h($benchmarkResult['version']) ?></td>
                                        </tr>
                                        <tr class="even">
                                            <td>Time</td>
                                            <td> <?php echo h($benchmarkResult['sysinfo']['time']) ?></td>
                                        </tr>
                                        <?php
                                        if (!empty($data['sysinfo']['xdebug']))
                                        {
                                            // You are running the benchmark with xdebug enabled. This has a major impact on runtime performance.
                                            echo '<tr class="even"><td>Xdebug</td><td><span style="color: darkred">'
                                                . h('Warning: Xdebug is enabled!')
                                                . '</span></td></tr>';
                                        }
                                        ?>
                                        <tr class="even">
                                            <td>PHP Version</td>
                                            <td><?php echo h($benchmarkResult['sysinfo']['php_version']) ?></td>
                                        </tr>
                                        <tr class="even">
                                            <td>Platform</td>
                                            <td><?php echo h($benchmarkResult['sysinfo']['platform']) ?></td>
                                        </tr>
                                        <?php
                                        if ($showServerName == true)
                                        {
                                            echo '<tr class="even"><td>Server name</td><td>' . h($benchmarkResult['sysinfo']['server_name']) . '</td></tr>' . "\n";
                                            echo '<tr class="even"><td>Server address</td><td>' . h($benchmarkResult['sysinfo']['server_addr']) . '</td></tr>' . "\n";
                                        }
                                        ?>
                                        </tbody>
                                    </table>
                                </div>
                                <div class="col-md-12">
                                    <table class="table table-sm table-striped">
                                        <thead>
                                        <tr>
                                            <th width="50%">Benchmark</th>
                                            <th width="50%"></th>
                                        </tr>
                                        </thead>
                                        <tbody>

                                        <tr>
                                            <td>Math</td>
                                            <td> <?php echo h($benchmarkResult['benchmark']['math']) ?></td>
                                        </tr>
                                        <tr>
                                            <td>String</td>
                                            <td><?php echo h($benchmarkResult['benchmark']['string']) ?> </td>
                                        </tr>
                                        <tr>
                                            <td>Loops</td>
                                            <td>  <?php echo h($benchmarkResult['benchmark']['loops']) ?></td>
                                        </tr>
                                        <tr>
                                            <td>If Else</td>
                                            <td> <?php echo h($benchmarkResult['benchmark']['ifelse']) ?> </td>
                                        </tr>
                                        <tr class="even">
                                            <td>Calculation total</td>
                                            <td><?php echo h($benchmarkResult['benchmark']['calculation']) ?></td>
                                        </tr>

                                        </tbody>
                                    </table>
                                </div>
                                <div class="col-md-12">
                                    <?php
                                    if (isset($benchmarkResult['sysinfo']['mysql_version']))
                                    {
                                        ?>
                                        <table class="table table-sm table-striped">
                                            <thead>
                                            <thead>
                                            <tr>
                                                <th width="50%">MySQL</th>
                                                <th width="50%"></th>
                                            </tr>
                                            </thead>
                                            </thead>
                                            <tbody>
                                            <tr>
                                                <td>MySQL Version</td>
                                                <td> <?php echo h($benchmarkResult['sysinfo']['mysql_version']) ?> </td>
                                            </tr>
                                            <tr>
                                                <td>MySQL Connect</td>
                                                <td> <?php echo h($benchmarkResult['benchmark']['mysql_connect']) ?> </td>
                                            </tr>
                                            <tr>
                                                <td>MySQL Select DB</td>
                                                <td> <?php echo h($benchmarkResult['benchmark']['mysql_select_db']) ?> </td>
                                            </tr>
                                            <tr>
                                                <td>MySQL Query Version</td>
                                                <td> <?php echo h($benchmarkResult['benchmark']['mysql_query_version']) ?></td>
                                            </tr>
                                            <tr>
                                                <td>MySQL Benchmark</td>
                                                <td><?php echo h($benchmarkResult['benchmark']['mysql_query_benchmark']) ?></td>
                                            </tr>
                                            <tr class="even">
                                                <td>MySQL Total</td>
                                                <td><?php echo h($benchmarkResult['benchmark']['mysql_total']) ?> </td>
                                            </tr>

                                            </tbody>
                                        </table>
                                        <?php
                                    }
                                    ?>

                                </div>
                                <div class="col-md-12">
                                    <table class="table table-sm table-striped mb-0">
                                        <thead>
                                        <tr>
                                            <th width="50%">Total</th>
                                            <th width="50%"><?php echo h($benchmarkResult['benchmark']['total']) ?></th>
                                        </tr>
                                        </thead>

                                    </table>
                                </div>
                                <div class="col-md-12">
                                    <table class="table table-sm table-striped">
                                        <tr>
                                            <td align="left"><strong>Recent Log:</strong></td>
                                        </tr>
                                        <tr>
                                            <td>
                         <textarea style="width: 100%; min-height: 300px;" readonly="readonly" disabled>
                                            <?php echo logReader('storage/logs/laravel.log') ?>
                                        </textarea>
                                            </td>
                                        </tr>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>
</div>
</body>
</html>