<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Reports\Reports;

use FI\Modules\Currencies\Models\Currency;
use FI\Modules\Expenses\Models\Expense;
use FI\Modules\Invoices\Models\Invoice;
use FI\Modules\Invoices\Models\InvoiceItem;
use FI\Modules\TaxRates\Models\TaxRate;
use FI\Support\CurrencyFormatter;
use FI\Support\DateFormatter;
use FI\Support\NumberFormatter;

class TaxReport
{
    public function formattedNumericAmount($amount)
    {
        return floatval(number_format($amount, 2, '.', ''));
    }
    public function getDetailResults($fromDate, $toDate, $excludeUnpaidInvoices = 0, $expense_type = null, $invoice_with_no_tax = 0, $group_by = 'item')
    {
        $results          = [
            'from_date'       => DateFormatter::format($fromDate),
            'to_date'         => DateFormatter::format($toDate),
            'grand_total_tax' => [],
            'records'         => [],
            'expenses'        => []
        ];
        $taxRateListNames = TaxRate::getListNames();
        $itemsTaxZero     = InvoiceItem::where('tax_rate_id', 0)->where('tax_rate_2_id', 0)->pluck('id')->toArray();
        $items            = InvoiceItem::byDateRange($fromDate, $toDate)
            ->select('invoice_items.name AS item_name', 'invoice_items.quantity AS item_quantity',
                'invoice_items.price AS item_price', 'clients.name AS client_name', 'invoices.number AS invoice_number',
                'invoices.invoice_date AS invoice_date', 'invoices.exchange_rate AS invoice_exchange_rate',
                'invoice_item_amounts.subtotal', 'invoice_item_amounts.tax', 'invoice_item_amounts.total',
                'invoice_amounts.discount', 'invoices.currency_code AS currency', 'invoice_items.tax_rate_id', 'invoice_items.tax_rate_2_id',
                'invoice_item_amounts.tax_1', 'invoice_item_amounts.tax_2')
            ->join('invoices', 'invoices.id', '=', 'invoice_items.invoice_id')
            ->join('invoice_item_amounts', 'invoice_item_amounts.item_id', '=', 'invoice_items.id')
            ->join('clients', 'clients.id', '=', 'invoices.client_id')
            ->join('invoice_amounts', 'invoice_items.invoice_id', '=', 'invoice_amounts.invoice_id')
            ->where('invoices.status', '<>', 'canceled')
            ->where(function ($q) use ($excludeUnpaidInvoices)
            {
                if ($excludeUnpaidInvoices)
                {
                    $q->where(function ($e)
                    {
                        $e->where('invoice_amounts.balance', '==', '0')
                            ->where('invoice_amounts.total', '>', '0');
                    })->where('invoices.status', '<>', 'canceled');
                }

            });

        if ($invoice_with_no_tax == 1)
        {
            $items->whereNotIn('invoice_items.id', $itemsTaxZero);
        }

        if ($group_by == 'invoice')
        {
            $items->orderBy('invoices.number');
        }
        else
        {
            $items->orderBy('invoice_items.name');
        }

        $items = $items->get();

        $invoiceCurrencies = Invoice::where('invoice_date', '>=', $fromDate)->where('invoice_date', '<=', $toDate)->groupBy('currency_code')->pluck('currency_code', 'currency_code')->toArray();

        foreach ($invoiceCurrencies as $currency)
        {
            $results['records'][$currency]         = [];
            $results['grand_total_tax'][$currency] = 0;
        }

        foreach ($items as $item)
        {

            $parentName = $item->item_name;
            $childName  = $item->invoice_number;

            if ($group_by == 'invoice')
            {
                $parentName = $item->invoice_number;
                $childName  = $item->item_name;
            }

            $currency                                              = $item->currency;
            $results['records'][$currency][$parentName]['items'][] = [
                'client_name'    => $item->client_name,
                'invoice_number' => $childName,
                'date'           => DateFormatter::format($item->invoice_date),
                'price'          => CurrencyFormatter::format($item->item_price),
                'quantity'       => NumberFormatter::format($item->item_quantity),
                'subtotal'       => CurrencyFormatter::format($item->subtotal),
                'discount'       => CurrencyFormatter::format($item->discount),
                'tax'            => $item->formattedTaxRate,
                'total_tax'      => CurrencyFormatter::format($item->tax),
            ];
            if (isset($results['records'][$currency][$parentName]['totals']))
            {
                $results['records'][$currency][$parentName]['totals']['quantity'] += $item->item_quantity;
                $results['records'][$currency][$parentName]['totals']['subtotal'] += round($item->subtotal, 2);
                $results['records'][$currency][$parentName]['totals']['discount'] += round($item->discount, 2);
                $results['records'][$currency][$parentName]['totals']['tax']      += round($item->tax, 2);
            }
            else
            {
                $results['records'][$currency][$parentName]['totals']['quantity'] = $item->item_quantity;
                $results['records'][$currency][$parentName]['totals']['subtotal'] = round($item->subtotal, 2);
                $results['records'][$currency][$parentName]['totals']['discount'] = round($item->discount, 2);
                $results['records'][$currency][$parentName]['totals']['tax']      = round($item->tax, 2);
            }

            foreach ($taxRateListNames as $taxName)
            {
                $results['records'][$currency][$parentName]['totals'][$taxName] = 0;
            }

        }

        foreach ($results['records'] as $code => $itemsArray)
        {
            foreach ($itemsArray as $itemName => $items)
            {
                foreach ($items['items'] as $item)
                {
                    foreach ($item['tax'] as $taxName => $itemTaxes)
                    {
                        if (isset($results['records'][$code][$itemName]['totals'][$taxName]))
                        {
                            $results['records'][$code][$itemName]['totals'][$taxName] += $this->formattedNumericAmount($itemTaxes);
                        }
                        else
                        {
                            $results['records'][$code][$itemName]['totals'][$taxName] = $this->formattedNumericAmount($itemTaxes);
                        }
                    }

                }

            }
        }

        foreach ($results['records'] as $code => $itemsArray)
        {
            $currency = Currency::whereCode($code)->first();

            foreach ($itemsArray as $key => $items)
            {
                if (isset($results['grand_total_tax'][$code]))
                {
                    $results['grand_total_tax'][$code] += $this->formattedNumericAmount($results['records'][$code][$key]['totals']['tax']);
                }
                else
                {
                    $results['grand_total_tax'][$code] = $this->formattedNumericAmount($results['records'][$code][$key]['totals']['tax']);
                }
                $results['records'][$code][$key]['totals']['quantity'] = NumberFormatter::format($results['records'][$code][$key]['totals']['quantity'], $currency);
                $results['records'][$code][$key]['totals']['subtotal'] = CurrencyFormatter::format($results['records'][$code][$key]['totals']['subtotal'], $currency);
                $results['records'][$code][$key]['totals']['discount'] = CurrencyFormatter::format($results['records'][$code][$key]['totals']['discount'], $currency);
                $results['records'][$code][$key]['totals']['tax']      = CurrencyFormatter::format($results['records'][$code][$key]['totals']['tax'], $currency);
                foreach ($taxRateListNames as $taxName)
                {
                    $results['records'][$code][$key]['totals'][$taxName] = CurrencyFormatter::format($results['records'][$code][$key]['totals'][$taxName], $currency);
                }

                foreach ($items['items'] as $index => $item)
                {
                    foreach ($item['tax'] as $taxName => $itemTaxes)
                    {
                        $results['records'][$code][$key]['items'][$index]['tax'][$taxName] = CurrencyFormatter::format($item['tax'][$taxName], $currency);;
                    }
                }
            }
        }

        foreach ($results['grand_total_tax'] as $code => $totalTax)
        {
            $currency                          = Currency::whereCode($code)->first();
            $results['grand_total_tax'][$code] = CurrencyFormatter::format($results['grand_total_tax'][$code], $currency);
        }

        $results['taxNames'] = $taxRateListNames;

        $results['expenses'] = $this->prepareExpenseRecords($fromDate, $toDate, $expense_type);

        return $results;
    }

    public function prepareExpenseRecords($fromDate, $toDate, $expense_type)
    {
        $expenses = Expense::dateRange($fromDate, $toDate)->type($expense_type)->where('type', '<>', 'standard_expense')->get();
        $records  = ['total' => 0, 'records' => [], 'tax' => []];
        foreach ($expenses as $expense)
        {
            $records['records'][$expense->id] =
                [
                    'expense_date'          => $expense->formatted_expense_date,
                    'vendor'                => $expense->formatted_vendor_name,
                    'vendor_id'             => $expense->vendor_id,
                    'client'                => isset($expense->client->name) ? $expense->client->name : '',
                    'category_name'         => $expense->formatted_category_name,
                    'formatted_description' => strip_tags($expense->formatted_description),
                    'description'           => $expense->description,
                    'formatted_tax_paid'    => $expense->formatted_tax_paid,
                    'tax_paid'              => $expense->tax_paid,
                    'type'                  => ($expense->type != '') ? trans('fi.' . $expense->type) : '',
                    'tax_name'              => isset($expense->taxRate->name) ? $expense->taxRate->name : '',
                ];
            $records['total']                 += $expense->tax_paid;
        }
        foreach ($records['records'] as $expense)
        {
            $tax_name = $expense['tax_name'];
            if (isset($records['tax'][$tax_name]))
            {
                $records['tax'][$tax_name] += floatval($expense['tax_paid']);
            }
            else
            {
                $records['tax'][$tax_name] = floatval($expense['tax_paid']);
            }
        }
        foreach ($records['tax'] as $name => $tax)
        {
            $records['tax'][$name] = CurrencyFormatter::format($tax);
        }
        $records['total'] = CurrencyFormatter::format($records['total']);
        return $records;
    }

    public function getSummaryResults($fromDate, $toDate, $companyProfileId = null, $excludeUnpaidInvoices = 0, $dateFilterBy, $invoice_with_no_tax = 0)
    {

        $results      = [
            'from_date' => DateFormatter::format($fromDate),
            'to_date'   => DateFormatter::format($toDate),
            'total'     => [],
            'paid'      => [],
            'remaining' => [],
            'records'   => [],
        ];
        $itemsTaxZero = InvoiceItem::where('tax_rate_id', 0)->where('tax_rate_2_id', 0)->pluck('id')->toArray();
        $invoices     = Invoice::select('invoices.*')->with(['items.taxRate', 'items.taxRate2', 'items.amount', 'amount'])
            ->join('invoice_amounts', 'invoices.id', '=', 'invoice_amounts.invoice_id')
            ->where(function ($q) use ($dateFilterBy, $fromDate, $toDate)
            {
                if ($dateFilterBy == 'invoice_date')
                {
                    $q->where('invoice_date', '>=', $fromDate)->where('invoice_date', '<=', $toDate);
                }
            })->where('status', '<>', 'canceled');

        if ($invoice_with_no_tax == 1)
        {
            $invoices->leftJoin('invoice_items', function ($join)
            {
                $join->on('invoice_items.invoice_id', '=', 'invoices.id');
            })->whereNotIn('invoice_items.id', $itemsTaxZero)->distinct();
        }

        if ($excludeUnpaidInvoices)
        {
            $invoices->leftJoin('payment_invoices', function ($join)
            {
                $join->on('payment_invoices.invoice_id', '=', 'invoices.id');
            })->join('payments', function ($join) use ($dateFilterBy, $fromDate, $toDate)
            {
                $join->on('payments.id', '=', 'payment_invoices.payment_id')->where(function ($q) use ($dateFilterBy, $fromDate, $toDate)
                {
                    if ($dateFilterBy == 'payment_date')
                    {
                        return $q->where('paid_at', '>=', $fromDate)->where('paid_at', '<=', $toDate);
                    }
                });
            });
        }

        if ($companyProfileId)
        {
            $invoices->where('company_profile_id', $companyProfileId);

            $expenseTax = (Expense::where('expense_date', '>=', $fromDate)
                ->where('expense_date', '<=', $toDate)
                ->where('company_profile_id', $companyProfileId)
                ->sum('tax_paid')) ?: 0;
        }
        else
        {
            $expenseTax = (Expense::where('expense_date', '>=', $fromDate)
                ->where('expense_date', '<=', $toDate)
                ->sum('tax_paid')) ?: 0;
        }

        if ($excludeUnpaidInvoices)
        {
            $invoices->paid();
        }

        $invoices = $invoices->get();

        foreach ($invoices as $invoice)
        {
            foreach ($invoice->items as $invoiceItem)
            {

                if ($invoiceItem->tax_rate_id)
                {
                    $key = $invoiceItem->taxRate->name . ' (' . NumberFormatter::format($invoiceItem->taxRate->percent, null, 3) . '%)';

                    if (isset($results['records'][$invoice->currency_code][$key]['taxable_amount']))
                    {
                        $results['records'][$invoice->currency_code][$key]['taxable_amount'] += $this->formattedNumericAmount($invoiceItem->amount->subtotal / $invoice->exchange_rate);
                        $results['records'][$invoice->currency_code][$key]['taxes']          += $this->formattedNumericAmount($invoiceItem->amount->tax_1 / $invoice->exchange_rate);
                    }
                    else
                    {
                        $results['records'][$invoice->currency_code][$key]['taxable_amount'] = $this->formattedNumericAmount($invoiceItem->amount->subtotal / $invoice->exchange_rate);
                        $results['records'][$invoice->currency_code][$key]['taxes']          = $this->formattedNumericAmount($invoiceItem->amount->tax_1 / $invoice->exchange_rate);
                    }
                }

                if ($invoiceItem->tax_rate_2_id)
                {
                    $key = $invoiceItem->taxRate2->name . ' (' . NumberFormatter::format($invoiceItem->taxRate2->percent, null, 3) . '%)';

                    if (isset($results['records'][$invoice->currency_code][$key]['taxable_amount']))
                    {
                        if ($invoiceItem->taxRate2->is_compound)
                        {
                            $results['records'][$invoice->currency_code][$key]['taxable_amount'] += $this->formattedNumericAmount(($invoiceItem->amount->subtotal + $invoiceItem->amount->tax_1) / $invoice->exchange_rate);
                        }
                        else
                        {
                            $results['records'][$invoice->currency_code][$key]['taxable_amount'] += $this->formattedNumericAmount($invoiceItem->amount->subtotal / $invoice->exchange_rate);
                        }

                        $results['records'][$invoice->currency_code][$key]['taxes'] += $this->formattedNumericAmount($invoiceItem->amount->tax_2 / $invoice->exchange_rate);
                    }
                    else
                    {
                        if ($invoiceItem->taxRate2->is_compound)
                        {
                            $results['records'][$invoice->currency_code][$key]['taxable_amount'] = $this->formattedNumericAmount(($invoiceItem->amount->subtotal + $invoiceItem->amount->tax_2) / $invoice->exchange_rate);
                        }
                        else
                        {
                            $results['records'][$invoice->currency_code][$key]['taxable_amount'] = $this->formattedNumericAmount($invoiceItem->amount->subtotal / $invoice->exchange_rate);
                        }

                        $results['records'][$invoice->currency_code][$key]['taxes'] = $this->formattedNumericAmount($invoiceItem->amount->tax_2 / $invoice->exchange_rate);
                    }
                }
            }
        }

        foreach ($results['records'] as $code => $records)
        {
            $currency = Currency::whereCode($code)->first();

            foreach ($records as $key => $result)
            {
                if (isset($results['total'][$code]))
                {
                    $results['total'][$code] = $this->formattedNumericAmount($results['total'][$code] + $result['taxes']);
                }
                else
                {
                    $results['total'][$code] = $this->formattedNumericAmount($result['taxes']);
                }
                $results['records'][$code][$key]['taxable_amount'] = CurrencyFormatter::format($result['taxable_amount'], $currency);
                $results['records'][$code][$key]['taxes']          = CurrencyFormatter::format($result['taxes'], $currency);
            }

            $results['paid'][$code]      = CurrencyFormatter::format($expenseTax, $currency);
            $results['remaining'][$code] = CurrencyFormatter::format($results['total'][$code] - $expenseTax, $currency);
            $results['total'][$code]     = CurrencyFormatter::format($results['total'][$code], $currency);
        }

        return $results;
    }
}