<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Reports\Controllers;

use FI\Http\Controllers\Controller;
use FI\Modules\CompanyProfiles\Models\CompanyProfile;
use FI\Modules\Mru\Events\MruLog;
use FI\Modules\Reports\Reports\TaxReport;
use FI\Modules\Reports\Requests\DateRangeRequest;
use FI\Support\PDF\PDFFactory;
use Illuminate\Support\Facades\Log;
use Throwable;

class TaxReportController extends Controller
{
    private $report;

    public function __construct(TaxReport $report)
    {
        $this->report = $report;
    }

    public function index()
    {
        $sort_option    = [
            'invoice_number' => trans('fi.filter_by_invoice_number'),
            'client_name'    => trans('fi.filter_by_client_name'),
            'invoice_date'   => trans('fi.filter_by_invoice_date'),
        ];
        $date_filter_by = [
            'invoice_date' => trans('fi.filter_by_invoice_date'),
            'payment_date' => trans('fi.filter_by_payment_date')
        ];
        return view('reports.options.tax_report')
            ->with('expenseType',
                [
                    ''                            => trans('fi.select_expense_type'),
                    'expense_with_deductible_tax' => trans('fi.expense_with_deductible_tax'),
                    'remittance_to_tax_authority' => trans('fi.remittance_to_tax_authority'),
                ])
            ->with('sort_option', $sort_option)
            ->with('date_filter_by', $date_filter_by);
    }

    public function validateOptions(DateRangeRequest $request)
    {

    }

    public function html()
    {
        event(new MruLog(['module' => 'reports', 'action' => 'tax_report', 'title' => trans('fi.tax_report'), 'id' => null]));

        if (request('level_of_detail') == 'detail')
        {
            $results = $this->report->getDetailResults(
                request('from_date'),
                request('to_date'),
                request('exclude_unpaid_invoices'),
                request('expense_type'),
                request('invoice_with_no_tax'),
                request('tax_group_by'),

            );

            $groupByText = trans('fi.invoice');
            if (request('tax_group_by') == 'invoice')
            {
                $groupByText = trans('fi.item');
            }

            return view('reports.output.tax_detail')
                ->with('results', $results)->with('logo', null)
                ->with('groupBy', $groupByText);
        }
        else
        {
            $results = $this->report->getSummaryResults(
                request('from_date'),
                request('to_date'),
                request('company_profile_id'),
                request('exclude_unpaid_invoices'),
                request('date_filter_by'),
                request('invoice_with_no_tax'),
            );

            $logo = null;
            if (request('company_profile_id'))
            {
                $companyProfile = CompanyProfile::whereId(request('company_profile_id'))->first();
                $logo['image']  = $companyProfile->logo(100);
                $logo['name']   = $companyProfile->company;
            }


            return view('reports.output.tax_summary')->with('logo', $logo)
                ->with('results', $results);
        }
    }

    public function pdf()
    {
        event(new MruLog(['module' => 'reports', 'action' => 'tax_report', 'title' => trans('fi.tax_report'), 'id' => null]));

        try
        {
            $pdf = PDFFactory::create();

            if (request('level_of_detail') == 'detail')
            {
                $pdf->setPaperOrientation('landscape');

                $results = $this->report->getDetailResults(
                    request('from_date'),
                    request('to_date'),
                    request('exclude_unpaid_invoices'),
                    request('expense_type'),
                    request('invoice_with_no_tax'),
                    request('tax_group_by'),
                );

                $logo = null;

                $groupByText = trans('fi.invoice');
                if (request('tax_group_by') == 'invoice')
                {
                    $groupByText = trans('fi.item');
                }

                $html = view('reports.output.tax_detail')->with('logo', $logo)->with('groupBy', $groupByText)
                    ->with('results', $results)->render();
                $html = (str_replace(iframeThemeColor(), '', $html));
                $pdf->download($html, trans('fi.tax_detail') . '.pdf');
            }
            else
            {
                $results = $this->report->getSummaryResults(
                    request('from_date'),
                    request('to_date'),
                    request('company_profile_id'),
                    request('exclude_unpaid_invoices'),
                    request('date_filter_by'),
                    request('invoice_with_no_tax'),
                );

                $logo = null;
                if (request('company_profile_id'))
                {
                    $companyProfile = CompanyProfile::whereId(request('company_profile_id'))->first();
                    $logo['image']  = $companyProfile->logo(100);
                    $logo['name']   = $companyProfile->company;
                }

                $html = view('reports.output.tax_summary')->with('logo', $logo)
                    ->with('results', $results)->render();
                $html = (str_replace(iframeThemeColor(), '', $html));
                $pdf->download($html, trans('fi.tax_summary') . '.pdf');
            }
        }
        catch (Throwable $e)
        {
            Log::error($e->getMessage());
        }

    }

    public function csv()
    {
        event(new MruLog(['module' => 'reports', 'action' => 'tax_detail', 'title' => trans('fi.tax_detail'), 'id' => null]));
        if (request('level_of_detail') == 'detail')
        {
            $results = $this->report->getDetailResults(
                request('from_date'),
                request('to_date'),
                request('exclude_unpaid_invoices'),
                request('expense_type'),
                request('invoice_with_no_tax'),
                request('tax_group_by'),
            );

            try
            {
                $headers     = [
                    "Content-type"        => "text/csv",
                    "Content-Disposition" => "attachment; filename=" . trans('fi.tax_detail') . '.csv',
                    "Pragma"              => "no-cache",
                    "Cache-Control"       => "must-revalidate, post-check=0, pre-check=0",
                    "Expires"             => "0"
                ];
                $groupByText = trans('fi.invoice');
                if (request('tax_group_by') == 'invoice')
                {
                    $groupByText = trans('fi.item');
                }
                $columns1                 = [
                    trans('fi.date'),
                    $groupByText,
                    trans('fi.client'),
                    trans('fi.price'),
                    trans('fi.quantity'),
                    trans('fi.subtotal'),
                    trans('fi.discount')
                ];
                $columns2                 = $results['taxNames'];
                $columns3                 = [
                    trans('fi.total'),
                ];
                $expensesColumns          = $this->expenseColumn($columns2);
                $columns                  = array_merge($columns1, $columns2, $columns3);
                $totalColumnForGrandTotal = count($columns) - 2;

                $callback = function () use ($results, $columns, $totalColumnForGrandTotal, $expensesColumns)
                {
                    $file = fopen('php://output', 'w');

                    fputcsv($file, ['', '', '', '', '', trans('fi.tax_detail'), '', '', '', '', '']);
                    fputcsv($file, ['', '', '', '', '', '', '', '', '', '', '']);
                    fputcsv($file, ['', '', '', '', '', trans('fi.taxes_collected_on_sales'), '', '', '', '', '']);
                    fputcsv($file, ['', '', '', '', $results['from_date'], '', $results['to_date'], '', '', '', '']);

                    if (count($results['records']) > 0)
                    {
                        foreach ($results['records'] as $currency => $itemsArray)
                        {
                            fputcsv($file, ['', '', '', '', '', '', '', '', '', '']);

                            $iterationCount = 0;

                            foreach ($itemsArray as $itemName => $items)
                            {
                                $iterationCount++;

                                if ($iterationCount === 1)
                                {
                                    fputcsv($file, ['', '', '', '', $currency, '', '', '', '']);
                                    fputcsv($file, $columns);
                                }
                                fputcsv($file, ['', '', '', '', '', '', '', '', '', '']);
                                fputcsv($file, ['', '', '', '', $itemName, '', '', '', '']);
                                foreach ($items['items'] as $item)
                                {
                                    $date           = $item['date'];
                                    $invoice_number = $item['invoice_number'];
                                    $client_name    = $item['client_name'];
                                    $price          = $item['price'];
                                    $quantity       = $item['quantity'];
                                    $subtotal       = $item['subtotal'];
                                    $discount       = $item['discount'];
                                    $total_tax      = $item['total_tax'];
                                    $taxes          = [];
                                    foreach ($results['taxNames'] as $taxName)
                                    {
                                        $taxes[$taxName] = isset($item['tax'][$taxName]) ? $item['tax'][$taxName] : '';
                                    }
                                    $merge = array_merge([$date, $invoice_number, $client_name, $price, $quantity, $subtotal, $discount], array_values($taxes), [$total_tax]);
                                    fputcsv($file, $merge);
                                }
                                $taxMain = [];
                                foreach ($results['taxNames'] as $taxName)
                                {
                                    $taxMain[$taxName] = isset($items['totals'][$taxName]) ? $items['totals'][$taxName] : '';
                                }
                                $mainMerge = array_merge(['', '', '', '', $items['totals']['quantity'], $items['totals']['subtotal'], $items['totals']['discount']], array_values($taxMain), [$items['totals']['tax']]);

                                fputcsv($file, $mainMerge);
                            }

                            $grandTotal = [];
                            for ($i = 0; $i < $totalColumnForGrandTotal; $i++)
                            {
                                $grandTotal[$i] = "";
                            }
                            $grandTotal[] = trans('fi.total');
                            $grandTotal[] = $results['grand_total_tax'][$currency];

                            fputcsv($file, $grandTotal);
                        }
                    }
                    else
                    {
                        fputcsv($file, ['', '', '', '', '', trans('fi.no_records_found'), '', '', '', '', '']);
                    }

                    if (count($results['expenses']['records']) > 0)
                    {
                        fputcsv($file, ['', '', '', '', '', '', '', '']);
                        fputcsv($file, ['', '', '', '', '', '', '', '']);
                        fputcsv($file, ['', '', '', '', '', '', '', '']);
                        fputcsv($file, ['', '', '', '', trans('fi.taxes_paid_on_expenses_or_remitted'), '', '', '', '']);
                        fputcsv($file, ['', '', '', $results['from_date'], '', $results['to_date'], '', '', '']);
                        fputcsv($file, ['', '', '', '', '', '', '', '']);
                        fputcsv($file, $expensesColumns);

                        foreach ($results['expenses']['records'] as $key => $expense)
                        {
                            $expense_date          = $expense['expense_date'];
                            $type                  = $expense['type'];
                            $vendor                = $expense['vendor'];
                            $client                = $expense['client'];
                            $category_name         = $expense['category_name'];
                            $formatted_description = $expense['formatted_description'];
                            $tax_name              = $expense['tax_name'];
                            $formatted_tax_paid    = $expense['formatted_tax_paid'];


                            $taxes = [];
                            foreach ($results['taxNames'] as $taxName)
                            {
                                $taxes[$taxName] = ($taxName == $tax_name) ? $formatted_tax_paid : '';
                            }
                            $expenseData = array_merge([
                                $expense_date,
                                $type,
                                $vendor,
                                $client,
                                $category_name,
                                $formatted_description,
                            ], array_values($taxes), [$formatted_tax_paid]);

                            fputcsv($file, $expenseData);
                        }

                        $mainTax = [];
                        foreach ($results['taxNames'] as $taxName)
                        {
                            $mainTax[$taxName] = (isset($results['expenses']['tax'][$taxName])) ? $results['expenses']['tax'][$taxName] : '';
                        }
                        fputcsv($file, array_merge(['', '', '', '', '', trans('fi.total')], array_values($mainTax), [$results['expenses']['total']]));
                    }
                    else
                    {
                        fputcsv($file, ['', '', '', '', '', '', '', '']);
                        fputcsv($file, ['', '', '', '', '', '', '', '']);
                        fputcsv($file, ['', '', '', '', trans('fi.taxes_paid_on_expenses_or_remitted'), '', '', '', '']);
                        fputcsv($file, ['', '', '', $results['from_date'], '', $results['to_date'], '', '', '']);
                        fputcsv($file, ['', '', '', '', '', '', '', '']);
                        fputcsv($file, ['', '', '', '', trans('fi.no_records_found'), '', '', '', '']);
                    }


                    fclose($file);
                };

                return response()->streamDownload($callback, trans('fi.tax_detail') . '.csv', $headers);
            }
            catch (Throwable $e)
            {
                Log::error($e->getMessage());
            }
        }
        else
        {
            $results = $this->report->getSummaryResults(
                request('from_date'),
                request('to_date'),
                request('company_profile_id'),
                request('exclude_unpaid_invoices'),
                request('date_filter_by'),
                request('invoice_with_no_tax'),
            );

            try
            {
                $headers = [
                    "Content-type"        => "text/csv",
                    "Content-Disposition" => "attachment; filename=" . trans('fi.tax_summary') . '.csv',
                    "Pragma"              => "no-cache",
                    "Cache-Control"       => "must-revalidate, post-check=0, pre-check=0",
                    "Expires"             => "0"
                ];

                $callback = function () use ($results)
                {
                    $file = fopen('php://output', 'w');

                    fputcsv($file, ['', $results['from_date'], '', $results['to_date']]);
                    fputcsv($file, ['', '', '']);
                    fputcsv($file, ['', trans('fi.tax_rate'), trans('fi.taxable_amount'), trans('fi.taxes')]);
                    foreach ($results['records'] as $currency => $records)
                    {
                        fputcsv($file, ['', '', $currency, '']);

                        foreach ($records as $taxRate => $result)
                        {
                            fputcsv($file, ['', $taxRate, $result['taxable_amount'], $result['taxes']]);
                        }

                        fputcsv($file, ['', '', '', '']);
                        fputcsv($file, ['', '', trans('fi.total'), $results['total'][$currency]]);
                        fputcsv($file, ['', '', trans('fi.paid'), $results['paid'][$currency]]);
                        fputcsv($file, ['', '', trans('fi.remaining'), $results['remaining'][$currency]]);
                    }
                    fclose($file);

                };

                return response()->streamDownload($callback, trans('fi.tax_summary') . '.csv', $headers);
            }
            catch (Throwable $e)
            {
                Log::error($e->getMessage());
            }

        }
    }

    public function expenseColumn($columns2)
    {
        $columns1 = [
            trans('fi.date'),
            trans('fi.type'),
            trans('fi.vendor'),
            trans('fi.client'),
            trans('fi.category'),
            trans('fi.description'),
        ];
        $columns3 = [
            trans('fi.total')
        ];

        return array_merge($columns1, $columns2, $columns3);
    }
}