<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\RecurringInvoices\Requests;

use FI\Modules\RecurringInvoices\Models\RecurringInvoice;
use FI\Requests\RemoveTagsRule;
use FI\Support\NumberFormatter;
use FI\Traits\CustomFieldValidator;

class RecurringInvoiceUpdateRequest extends RecurringInvoiceStoreRequest
{
    use CustomFieldValidator;

    private $customFieldType         = 'recurring_invoices';
    private $lineItemCustomFieldType = 'recurring_invoice_items';

    public function prepareForValidation()
    {
        $request  = $this->all();
        $ri       = RecurringInvoice::find($request['recurring_invoice_id']);
        $currency = isset($ri->currency_code) ? $ri->currency : NULL;

        if (isset($request['items']))
        {
            foreach ($request['items'] as $key => $item)
            {
                $request['items'][$key]['quantity'] = NumberFormatter::unformat($item['quantity'], $currency);
                $request['items'][$key]['price']    = NumberFormatter::unformat($item['price'], $currency);
            }
        }

        $this->replace($request);
    }

    public function rules()
    {
        return [
            'summary'             => ['max:255', new RemoveTagsRule],
            'exchange_rate'       => 'required|numeric',
            'template'            => 'required',
            'items'               => 'required',
            'items.*.name'        => 'required',
            'items.*.quantity'    => 'required_with:items.*.price,items.*.name|numeric',
            'items.*.price'       => 'required_with:items.*.name,items.*.quantity|numeric',
            'items.*.description' => [new RemoveTagsRule],
            'terms'               => [new RemoveTagsRule],
            'footer'              => [new RemoveTagsRule],
            'number'              => [new RemoveTagsRule],
            'payment_terms_id'    => 'required|exists:payment_terms,id',
            'discount'            => function ($attribute, $value, $fail)
            {
                if ($this->discount_type == 'percentage')
                {
                    if ($value > 99.99)
                    {
                        $fail(trans('fi.discount_limit'));
                    }
                }
            },
            'tags.*'              => [new RemoveTagsRule]
        ];
    }

    public function messages()
    {
        return [
            'items.*.price' => trans('fi.price_not_valid'),
        ];
    }


}