<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Quotes\Requests;

use FI\Modules\Quotes\Models\Quote;
use FI\Requests\RemoveTagsRule;
use FI\Support\NumberFormatter;
use FI\Traits\CustomFieldValidator;
use Illuminate\Validation\Rule;

class QuoteUpdateRequest extends QuoteStoreRequest
{
    use CustomFieldValidator;

    private $customFieldType         = 'quotes';
    private $lineItemCustomFieldType = 'quote_items';

    public function attributes()
    {
        return [
            'is_payable_quote' => trans('fi.allow_payments_on_quote'),
        ];
    }

    public function prepareForValidation()
    {
        $request  = $this->all();
        $quote    = Quote::find($request['quote_id']);
        $currency = isset($quote->currency_code) ? $quote->currency : NULL;
        if (isset($request['items']))
        {
            foreach ($request['items'] as $key => $item)
            {
                $request['items'][$key]['quantity'] = NumberFormatter::unformat($item['quantity'], $currency);
                $request['items'][$key]['price']    = NumberFormatter::unformat($item['price'], $currency);

                if (isset($request['items'][$key]['discount']))
                {
                    $request['items'][$key]['discount'] = NumberFormatter::unformat($item['discount'], $currency);
                }
            }
        }

        $this->replace($request);
    }

    public function rules()
    {
        return [
            'is_payable_quote'    => Rule::requiredIf(fn() => config('fi.isPayableQuote') == 1),
            'summary'             => ['max:255', new RemoveTagsRule],
            'payment_term_id'     => 'required|exists:payment_terms,id',
            'quote_date'          => 'required',
            'number'              => ['required', new RemoveTagsRule],
            'status'              => 'required',
            'exchange_rate'       => 'required|numeric',
            'template'            => 'required',
            'items'               => 'required',
            'items.*.name'        => 'required',
            'items.*.quantity'    => 'required_with:items.*.price,items.*.name|numeric',
            'items.*.price'       => 'required_with:items.*.name,items.*.quantity|numeric',
            'items.*.description' => [new RemoveTagsRule],
            'terms'               => [new RemoveTagsRule],
            'footer'              => [new RemoveTagsRule],
            'items.*.discount'    => function ($attribute, $value, $fail)
            {
                $index = explode('.', $attribute);
                if (isset($this['items'][$index[1]]['discount_type']) && $this['items'][$index[1]]['discount_type'] == 'percentage')
                {
                    if ($value > 99.99)
                    {
                        $fail(trans('fi.line_item_discount_limit'));
                    }
                    return true;
                }
                return true;
            },
            'discount'            => function ($attribute, $value, $fail)
            {
                if ($this->discount_type == 'percentage')
                {
                    if ($value > 99.99)
                    {
                        $fail(trans('fi.discount_limit'));
                    }
                }
            },
            'tags.*'              => [new RemoveTagsRule]
        ];
    }

    public function messages()
    {
        return [
            'items.*.price' => trans('fi.price_not_valid'),
        ];
    }
}