<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Quotes\Models;

use Carbon\Carbon;
use FI\Modules\CustomFields\Support\CustomFieldsParser;
use FI\Modules\PaymentTerms\Models\PaymentTermInstallment;
use FI\Modules\Quotes\Support\QuotePaymentTerm;
use FI\Modules\Tags\Models\Tag;
use FI\Support\CurrencyFormatter;
use FI\Support\DateFormatter;
use FI\Support\FileNames;
use FI\Support\HTML;
use FI\Support\NumberFormatter;
use FI\Traits\Sortable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use stdClass;

class Quote extends Model
{
    use Sortable;

    protected $guarded = ['id'];

    protected $sortable = [
        'number' => ['LENGTH(number)', 'number'],
        'quote_date',
        'expires_at',
        'clients.name',
        'summary',
        'quote_amounts.total',
        'quote_amounts.tax',
        'quote_amounts.subtotal',
    ];

    protected $dates = ['expires_at', 'quote_date'];

    /*
    |--------------------------------------------------------------------------
    | Relationships
    |--------------------------------------------------------------------------
    */

    public function activities()
    {
        return $this->morphMany('FI\Modules\Activity\Models\Activity', 'audit');
    }

    public function revision()
    {
        return $this->morphMany('FI\Modules\DocumentRevisions\Models\DocumentRevision', 'revisable');
    }

    public function amount()
    {
        return $this->hasOne('FI\Modules\Quotes\Models\QuoteAmount');
    }

    public function attachments()
    {
        return $this->morphMany('FI\Modules\Attachments\Models\Attachment', 'attachable');
    }

    public function tags()
    {
        return $this->hasMany('FI\Modules\Quotes\Models\QuoteTag');
    }

    public function client()
    {
        return $this->belongsTo('FI\Modules\Clients\Models\Client');
    }

    public function clientAttachments()
    {
        $relationship = $this->morphMany('FI\Modules\Attachments\Models\Attachment', 'attachable');

        $relationship->where('client_visibility', 1);

        return $relationship;
    }

    public function companyProfile()
    {
        return $this->belongsTo('FI\Modules\CompanyProfiles\Models\CompanyProfile');
    }

    public function currency()
    {
        return $this->belongsTo('FI\Modules\Currencies\Models\Currency', 'currency_code', 'code');
    }

    public function custom()
    {
        return $this->hasOne('FI\Modules\CustomFields\Models\QuoteCustom');
    }

    public function documentNumberScheme()
    {
        return $this->hasOne('FI\Modules\DocumentNumberSchemes\Models\DocumentNumberScheme');
    }

    public function invoice()
    {
        return $this->belongsTo('FI\Modules\Invoices\Models\Invoice');
    }

    public function mailQueue()
    {
        return $this->morphMany('FI\Modules\MailQueue\Models\MailQueue', 'mailable');
    }

    public function smsQueue()
    {
        return $this->morphMany('Addons\Sms\Models\SmsQueue', 'mailable');
    }

    public function items()
    {
        return $this->hasMany('FI\Modules\Quotes\Models\QuoteItem')
            ->orderBy('display_order');
    }

    public function notifications()
    {
        return $this->morphMany('FI\Modules\Notifications\Models\Notification', 'notifiable');
    }

    public function notes()
    {
        return $this->morphMany('FI\Modules\Notes\Models\Note', 'notable');
    }

    // This and items() are the exact same. This is added to appease the IDE gods
    // and the fact that Laravel has a protected items property.
    public function quoteItems()
    {
        return $this->hasMany('FI\Modules\Quotes\Models\QuoteItem')
            ->orderBy('display_order');
    }

    public function transitions()
    {
        return $this->morphMany('FI\Modules\Transitions\Models\Transitions', 'transitionable');
    }

    public function user()
    {
        return $this->belongsTo('FI\Modules\Users\Models\User');
    }

    public function tagName()
    {
        return $this->belongsToMany(Tag::class, 'quote_tags');
    }

    public function paymentTerm()
    {
        return $this->belongsTo('FI\Modules\PaymentTerms\Models\PaymentTerm', 'payment_term_id', 'id');
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors
    |--------------------------------------------------------------------------
    */
    public function getFormattedTagsAttribute()
    {
        $quoteTags = [];

        foreach ($this->tags as $tag)
        {
            $quoteTags[] = $tag->tag->name;
        }

        if (empty($quoteTags))
        {
            return '';
        }
        else
        {
            if (count($quoteTags) == 1)
            {
                return $quoteTags[0];
            }
            else if (count($quoteTags) == 2)
            {
                return $quoteTags[0] . ', ' . $quoteTags[1];
            }
            else
            {
                return $quoteTags[0] . ', ' . $quoteTags[1] . '..';
            }

        }

    }

    public function getAttachmentPermissionOptionsAttribute()
    {
        return ['0' => trans('fi.not_visible'), '1' => trans('fi.visible')];
    }

    public function getFormattedCreatedAtAttribute()
    {
        return $this->formatted_quote_date;
    }

    public function getFormattedQuoteDateAttribute()
    {
        return DateFormatter::format($this->attributes['quote_date']);
    }

    public function getFormattedUpdatedAtAttribute()
    {
        return DateFormatter::format($this->attributes['updated_at']);
    }

    public function getFormattedExpiresAtAttribute()
    {
        return DateFormatter::format($this->attributes['expires_at']);
    }

    public function getFormattedTermsAttribute()
    {
        return nl2br($this->attributes['terms']);
    }

    public function getFormattedFooterAttribute()
    {
        return nl2br($this->attributes['footer']);
    }

    public function getStatusTextAttribute()
    {
        return $this->attributes['status'];
    }

    public function getPdfFilenameAttribute()
    {
        return FileNames::quote($this);
    }

    public function getPublicUrlAttribute()
    {
        return route('clientCenter.public.quote.show', [$this->url_key, $this->token]);
    }

    public function getIsForeignCurrencyAttribute()
    {
        if ($this->attributes['currency_code'] == config('fi.baseCurrency'))
        {
            return false;
        }

        return true;
    }

    public function getHtmlAttribute()
    {
        return HTML::quote($this);
    }

    public function getFormattedNumericDiscountAttribute()
    {
        return NumberFormatter::format($this->attributes['discount']);
    }

    public function getTokenAttribute()
    {
        return ((config('fi.secure_link')) == 1) ? Crypt::encrypt(Carbon::now()->addDay(config('fi.secure_link_expire_day'))->format('Y-m-d')) : null;
    }

    /**
     * Gathers a summary of both invoice and item taxes to be displayed on invoice.
     *
     * @return array
     */
    public function getSummarizedTaxesAttribute()
    {
        $taxes = [];

        foreach ($this->items as $item)
        {
            if ($item->taxRate)
            {
                $key = $item->taxRate->name;

                if (!isset($taxes[$key]))
                {
                    $taxes[$key]              = new stdClass();
                    $taxes[$key]->name        = $item->taxRate->name;
                    $taxes[$key]->percent     = $item->taxRate->formatted_percent;
                    $taxes[$key]->total       = $item->amount->tax_1;
                    $taxes[$key]->raw_percent = $item->taxRate->percent;
                }
                else
                {
                    $taxes[$key]->total += $item->amount->tax_1;
                }
            }

            if ($item->taxRate2)
            {
                $key = $item->taxRate2->name;

                if (!isset($taxes[$key]))
                {
                    $taxes[$key]              = new stdClass();
                    $taxes[$key]->name        = $item->taxRate2->name;
                    $taxes[$key]->percent     = $item->taxRate2->formatted_percent;
                    $taxes[$key]->total       = $item->amount->tax_2;
                    $taxes[$key]->raw_percent = $item->taxRate2->percent;
                }
                else
                {
                    $taxes[$key]->total += $item->amount->tax_2;
                }
            }
        }

        foreach ($taxes as $key => $tax)
        {
            $taxes[$key]->total = CurrencyFormatter::format($tax->total, $this->currency, config('fi.roundTaxDecimals'));
        }

        return $taxes;
    }


    public function getShortSummaryAttribute()
    {
        return (mb_strlen($this->summary) > 50) ? mb_substr($this->summary, 0, 50) . '...' : $this->summary;
    }

    public function hasLineItemDiscount()
    {
        $hasDiscount = false;

        foreach ($this->items as $item)
        {
            if ($item->amount->discount_amount != 0)
            {
                $hasDiscount = true;
                break;
            }
        }

        return $hasDiscount;
    }

    public function getTagNamesAttribute()
    {
        return $this->tagName->pluck('name')->toArray();
    }

    public function getFormattedPaymentTermDueDateAttribute()
    {
        $paymentTerm = $this->paymentTerm;

        if ($paymentTerm->type == 0)
        {
            return ['expire' => $this->expires_at];
        }
        elseif ($paymentTerm->type == 2)
        {
            return ['expire' => Carbon::createFromDate($this->quote_date)->addDays($paymentTerm->discount_days_simple)->format('Y-m-d')];
        }
        elseif ($paymentTerm->type == 3)
        {
            $type3ExpireDate = Carbon::createFromDate($this->quote_date)->addMonth()->startOfMonth()->addDays($paymentTerm->discount_days_nth)->subDay()->format('Y-m-d');
            return ['expire' => $type3ExpireDate];
        }
        elseif ($paymentTerm->type == 4)
        {
            return ['expire' => Carbon::createFromDate($this->quote_date)->addDays($paymentTerm->balance_due_days)->format('Y-m-d')];
        }
        elseif ($paymentTerm->type == 5)
        {
            $paymentTermInstallments    = PaymentTermInstallment::wherePaymentTermsId($paymentTerm->id)->orderBy('installment_number', 'ASC')->get();
            $lastPaymentTermInstallment = PaymentTermInstallment::wherePaymentTermsId($paymentTerm->id)->whereInstallmentNumber($paymentTerm->number_of_installments)->first();

            if (count($paymentTermInstallments) > 0)
            {

                foreach ($paymentTermInstallments as $paymentTermInstallment)
                {
                    $expireDateOfOffer     = Carbon::createFromDate($this->quote_date)->addDays($paymentTermInstallment->installment_due_days)->format('Y-m-d');
                    $lastExpireDateOfOffer = Carbon::createFromDate($this->quote_date)->addDays($lastPaymentTermInstallment->installment_due_days)->format('Y-m-d');

                    return array_merge($paymentTermInstallment->toArray() + ['expire' => $lastExpireDateOfOffer, 'installment_expire' => $expireDateOfOffer, 'lastPayment' => ($paymentTermInstallment->id == $lastPaymentTermInstallment->id) ? true : false]);
                }
            }
        }
        else
        {
            return ['expire' => Carbon::createFromDate($this->quote_date)->addDays($paymentTerm->net_due_days)->format('Y-m-d')];
        }
    }

    public function getFormattedActualTermsTextAttribute()
    {
        $paymentTerm = $this->paymentTerm;

        if ($paymentTerm && $paymentTerm->quote_description_template != '')
        {
            $quotePaymentTerm = new QuotePaymentTerm();

            switch ($paymentTerm->type)
            {
                case 1:
                    return $quotePaymentTerm->processTypeOne($paymentTerm, $this);
                case 2:
                    return $quotePaymentTerm->processTypeTwo($paymentTerm, $this);
                case 3:
                    return $quotePaymentTerm->processTypeThree($paymentTerm, $this);
                case 4:
                    return $quotePaymentTerm->processTypeFour($paymentTerm, $this);
                case 5:
                    return $quotePaymentTerm->processTypeFive($paymentTerm, $this);
            }
        }
        return $paymentTerm->name ?? '';
    }

    /*
    |--------------------------------------------------------------------------
    | Static Methods
    |--------------------------------------------------------------------------
    */
    public static function getDiscountTypes()
    {
        return ['' => '&#xf05e;', 'percentage' => '&#xf295;', 'flat_amount' => '&#xf0d6;'];
    }

    public static function getHeaders()
    {
        return ['status', 'quote', 'date', 'expires', 'client', 'summary', 'invoiced', 'total'];
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */
    public function scopeDateRange($query, $fromDate = null, $toDate = null)
    {
        if ($fromDate != '' && $toDate != '')
        {
            $query->where('quotes.quote_date', '>=', $fromDate)->where('quotes.quote_date', '<=', $toDate);
        }

    }

    public function scopeTags($query, $tags, $tagsMustMatchAll)
    {
        if (!empty($tags))
        {
            if ($tagsMustMatchAll)
            {
                $query->whereHas('tags', function ($query) use ($tags)
                {
                    $query->whereIn("tag_id", $tags);

                }, "=", count($tags));
            }
            else
            {
                $query->whereHas('tags', function ($query) use ($tags)
                {
                    $query->whereIn("tag_id", $tags);

                });
            }
        }
    }

    public function scopeClient($query, $clientId = null)
    {
        if ($clientId)
        {
            $query->where('client_id', $clientId);
        }

        return $query;
    }

    public function scopeCompanyProfileId($query, $companyProfileId)
    {
        if ($companyProfileId)
        {
            $query->where('quotes.company_profile_id', $companyProfileId);
        }

        return $query;
    }

    public function scopeDraft($query)
    {
        return $query->where('status', '=', 'draft');
    }

    public function scopeSent($query)
    {
        return $query->where('status', '=', 'sent');
    }

    public function scopeApproved($query)
    {
        return $query->where('status', '=', 'approved');
    }

    public function scopeRejected($query)
    {
        return $query->where('status', '=', 'rejected');
    }

    public function scopeCanceled($query)
    {
        return $query->where('status', '=', 'canceled');
    }

    public function scopeStatus($query, $status = null)
    {
        switch ($status)
        {
            case 'draft':
                $query->draft();
                break;
            case 'sent':
                $query->sent();
                break;
            case 'viewed':
                $query->viewed();
                break;
            case 'approved':
                $query->approved();
                break;
            case 'rejected':
                $query->rejected();
                break;
            case 'canceled':
                $query->canceled();
                break;
        }

        return $query;
    }

    public function scopeThisYear($query)
    {
        return $query->where(DB::raw('YEAR(quote_date)'), '=', DB::raw('YEAR(CURRENT_DATE())'));
    }

    public function scopeThisMonth($query)
    {
        return $query->where(DB::raw('MONTH(quote_date)'), '=', DB::raw('MONTH(CURRENT_DATE())'))
            ->where(DB::raw('YEAR(quote_date)'), '=', DB::raw('YEAR(CURRENT_DATE())'));
    }

    public function scopeThisQuarter($query)
    {
        return $query->where('quote_date', '>=', Carbon::now()->firstOfQuarter())
            ->where('quote_date', '<=', Carbon::now()->lastOfQuarter());
    }

    public function scopeLastMonth($query)
    {
        return $query->where(DB::raw('quote_date'), '>=', Carbon::now()->subMonths(1)->firstOfMonth())
            ->where(DB::raw('quote_date'), '<=', Carbon::now()->subMonths(1)->lastOfMonth());
    }

    public function scopeLastQuarter($query)
    {
        return $query->where('quote_date', '>=', Carbon::now()->subQuarters(1)->firstOfQuarter())
            ->where('quote_date', '<=', Carbon::now()->subQuarters(1)->lastOfQuarter());
    }

    public function scopeLastYear($query)
    {
        return $query->where('quote_date', '>=', Carbon::now()->subYears(1)->firstOfYear())
            ->where('quote_date', '<=', Carbon::now()->subYears(1)->lastOfYear());
    }

    public function scopeToday($query)
    {
        return $query->where('quotes.quote_date', '=', Carbon::now()->format('Y-m-d'));
    }

    public function scopeYesterday($query)
    {
        return $query->where('quotes.quote_date', '=', Carbon::yesterday()->format('Y-m-d'));
    }

    public function scopeLast7Days($query)
    {
        return $query->where('quotes.quote_date', '>=', Carbon::now()->subDays(6)->format('Y-m-d'))
            ->where('quotes.quote_date', '<=', Carbon::now()->format('Y-m-d'));
    }

    public function scopeLast30Days($query)
    {
        return $query->where('quotes.quote_date', '>=', Carbon::now()->subDays(29)->format('Y-m-d'))
            ->where('quotes.quote_date', '<=', Carbon::now()->format('Y-m-d'));
    }

    public function scopeFirstQuarter($query)
    {

        $currentDate = (Carbon::createFromDate(Carbon::now()->format('Y'))->firstOfYear()->format('Y-m-d'));
        return $query->where('quote_date', '>=', Carbon::createFromDate($currentDate)->addQuarter(0)->startOf('quarter')->format('Y-m-d'))
            ->where('quote_date', '<=', Carbon::createFromDate($currentDate)->addQuarter(0)->endOf('quarter')->format('Y-m-d'));
    }

    public function scopeSecondQuarter($query)
    {
        $currentDate = (Carbon::createFromDate(Carbon::now()->format('Y'))->firstOfYear()->format('Y-m-d'));
        return $query->where('quote_date', '>=', Carbon::createFromDate($currentDate)->addQuarter(1)->startOf('quarter')->format('Y-m-d'))
            ->where('quote_date', '<=', Carbon::createFromDate($currentDate)->addQuarter(1)->endOf('quarter')->format('Y-m-d'));
    }

    public function scopeThirdQuarter($query)
    {
        $currentDate = (Carbon::createFromDate(Carbon::now()->format('Y'))->firstOfYear()->format('Y-m-d'));
        return $query->where('quote_date', '>=', Carbon::createFromDate($currentDate)->addQuarter(2)->startOf('quarter')->format('Y-m-d'))
            ->where('quote_date', '<=', Carbon::createFromDate($currentDate)->addQuarter(2)->endOf('quarter')->format('Y-m-d'));
    }

    public function scopeFourthQuarter($query)
    {
        $currentDate = (Carbon::createFromDate(Carbon::now()->format('Y'))->firstOfYear()->format('Y-m-d'));
        return $query->where('quote_date', '>=', Carbon::createFromDate($currentDate)->addQuarter(3)->startOf('quarter')->format('Y-m-d'))
            ->where('quote_date', '<=', Carbon::createFromDate($currentDate)->addQuarter(3)->endOf('quarter')->format('Y-m-d'));
    }

    public function scopeKeywords($query, $keywords)
    {
        if ($keywords)
        {
            $keywords                 = strtolower($keywords);
            $currencyStringMatchScore = currencyStringMatch($keywords);

            switch ($currencyStringMatchScore)
            {
                case 0:
                    // Definitely not a currency value search
                    $query->where(DB::raw('lower(number)'), 'like', '%' . $keywords . '%')
                        ->orWhere('quote_items.name', 'like', '%' . $keywords . '%')
                        ->orWhere('quote_items.description', 'like', '%' . $keywords . '%')
                        ->orWhere(function ($query) use ($keywords)
                        {
                            $formattedDate = DateFormatter::format($keywords);
                            if ($formattedDate !== null)
                            {
                                $date = Carbon::parse($formattedDate)->format('Y-m-d');
                                $query->where('quotes.quote_date', 'like', '%' . $date . '%')
                                    ->orWhere('quotes.expires_at', 'like', '%' . $date . '%');

                            }
                        })
                        ->orWhere('summary', 'like', '%' . $keywords . '%')
                        ->orWhereIn('client_id', function ($query) use ($keywords)
                        {
                            $query->select('id')->from('clients')->where(DB::raw("CONCAT_WS('^',LOWER(name))"), 'like', '%' . $keywords . '%');
                        });
                    break;

                case 1:
                    // *Could* be interpreted as a currency value, include the currency appropriate fields as well as other fields that might have a decimal number in them.
                    $decimalValue = convertToDecimalFormat($keywords);

                    $query->where(DB::raw('lower(number)'), 'like', '%' . $keywords . '%')
                        ->orWhere('summary', 'like', '%' . $keywords . '%')
                        ->orWhereHas('amount', function ($query) use ($decimalValue)
                        {
                            $query->whereRaw("REPLACE(FORMAT(quote_amounts.total, 2),',','') = $decimalValue");
                        });
                    break;

                case 2:
                    // String value started with a currency symbol, ONLY search the currency appropriate fields.
                    $currency     = getCurrencyFetch($keywords);
                    $decimalValue = convertToDecimalFormat(substr($keywords, 1));

                    $query = $query->whereIn('quotes.currency_code', $currency)
                        ->whereHas('amount', function ($query) use ($decimalValue)
                        {
                            $query->whereRaw("REPLACE(FORMAT(quote_amounts.total, 2),',','') = $decimalValue");
                        });
                    break;
            }
        }
        return $query;
    }

    public function scopeCustomField($query, $includeCustomFields = 0)
    {
        if ($includeCustomFields == 1)
        {
            $query->with('custom');
        }

        return $query;
    }

    public function scopeFieldsWiseSearch($query, $fieldsWiseSearch, $operator)
    {

        if ($fieldsWiseSearch != null)
        {
            if (count($fieldsWiseSearch) == 1)
            {
                $operator = 'and';
            }
            $query->where(function ($query) use ($fieldsWiseSearch, $operator)
            {
                foreach ($fieldsWiseSearch as $key => $value)
                {
                    if (substr($value, 0, 2) == '!=')
                    {
                        $query->whereNotIn($key, [substr($value, 2)]);
                    }
                    else
                    {
                        if ($operator == 'or')
                        {
                            if (substr_count($key, '->') == 0)
                            {
                                $query->orWhere('quotes.' . $key, 'like', '%' . $value . '%');
                            }
                            else
                            {
                                $tableAndField = explode('->', $key);

                                $query->whereHas($tableAndField[0], function ($query) use ($value, $tableAndField)
                                {
                                    $query->where($tableAndField[1], 'like', '%' . $value . '%');
                                });
                            }
                        }
                        if ($operator == 'and')
                        {
                            if (substr_count($key, '->') == 0)
                            {
                                $query->where('quotes.' . $key, 'like', '%' . $value . '%');
                            }
                            else
                            {
                                $tableAndField = explode('->', $key);

                                $query->whereHas($tableAndField[0], function ($query) use ($value, $tableAndField)
                                {
                                    $query->where($tableAndField[1], 'like', '%' . $value . '%');
                                });
                            }
                        }
                    }
                }
            });
        }
        return $query;
    }

    /*
    |--------------------------------------------------------------------------
    | Other
    |--------------------------------------------------------------------------
    */

    public function customField($label, $rawHtml = true)
    {
        $customField = config('fi.customFields')->where('tbl_name', 'quotes')->where('field_label', $label)->first();

        if ($customField)
        {
            return CustomFieldsParser::getFieldValue($this->custom, $customField, $rawHtml);
        }

        return null;
    }

    public function deleteTags(Quote $quote)
    {
        $quote->tags()->delete();
    }

    public function getPaymentDetailMessageAttribute()
    {
        $message = ['charge' => '', 'term' => ''];
        try
        {
            $invoiceAmountBalance       = CurrencyFormatter::format($this->amount->total, $this->currency);
            $onlinePaymentChargesDetail = onlinePaymentQuoteChargesDetail($this);
            $type                       = $onlinePaymentChargesDetail['type'];
            $amountValue                = CurrencyFormatter::format($onlinePaymentChargesDetail['amountValue'], $this->currency);

            if ($onlinePaymentChargesDetail['addChargesItem'])
            {
                $amountValue       = CurrencyFormatter::format($onlinePaymentChargesDetail['amountValue'] - $onlinePaymentChargesDetail['feeCharges'], $this->currency);
                $message['charge'] = trans('fi.credit_card_fee_message', ['charge' => CurrencyFormatter::format($onlinePaymentChargesDetail['feeCharges'], $this->currency)]);
            }

            if ($this->amount->total != $onlinePaymentChargesDetail['amountValue'] || in_array($type, [2, 3, 4, 5]))
            {
                if ($type == 2 || $type == 3)
                {
                    $message['term'] = trans('fi.info_payment_term_type_' . $type, ['ORIGINAL_AMOUNT' => $invoiceAmountBalance, 'DISCOUNT_PERCENT' => $onlinePaymentChargesDetail['percentage'], 'DISCOUNT_AMOUNT' => $amountValue]);
                }
                elseif ($type == 4 && $this->amount->total > $onlinePaymentChargesDetail['amountValue'])
                {
                    $message['term'] = trans('fi.info_payment_term_type_' . $type, ['ORIGINAL_AMOUNT' => $invoiceAmountBalance, 'DISCOUNT_AMOUNT' => $amountValue]);
                }
                elseif ($type == 5)
                {
                    $message['term'] = trans('fi.info_payment_term_type_' . $type, ['ORIGINAL_AMOUNT' => $invoiceAmountBalance, 'DISCOUNT_AMOUNT' => $amountValue]);
                }
            }

            if (empty(floatval($onlinePaymentChargesDetail['paymentTermsDiscount'])))
            {
                $message['term'] = '';
            }

            return $message;
        }
        catch (\Exception|\Error $e)
        {
            return $message;
        }
    }

}
