<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Quotes\Controllers;

use FI\Http\Controllers\Controller;
use FI\Modules\Currencies\Models\Currency;
use FI\Modules\CustomFields\Models\QuoteCustom;
use FI\Modules\CustomFields\Models\QuoteItemCustom;
use FI\Modules\CustomFields\Support\CustomFieldsParser;
use FI\Modules\CustomFields\Support\CustomFieldsTransformer;
use FI\Modules\DocumentRevisions\Support\DocumentRevisionSupport;
use FI\Modules\ItemLookups\Models\ItemLookup;
use FI\Modules\Mru\Events\MruLog;
use FI\Modules\PaymentTerms\Models\PaymentTerm;
use FI\Modules\Quotes\Events\AddTransition;
use FI\Modules\Quotes\Events\QuoteToInvoiceTransition;
use FI\Modules\Quotes\Models\Quote;
use FI\Modules\Quotes\Models\QuoteItem;
use FI\Modules\Quotes\Requests\QuoteUpdateRequest;
use FI\Modules\Quotes\Requests\QuoteUpdateSummaryAndTagRequest;
use FI\Modules\Quotes\Support\QuotePaymentTerm;
use FI\Modules\Quotes\Support\QuoteTemplates;
use FI\Modules\Quotes\Support\QuoteToInvoice;
use FI\Modules\Tags\Models\Tag;
use FI\Modules\TaxRates\Models\TaxRate;
use FI\Support\DateFormatter;
use FI\Support\Statuses\QuoteStatuses;
use FI\Traits\ReturnUrl;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class QuoteEditController extends Controller
{
    use ReturnUrl;

    public function edit($id)
    {
        $quote = Quote::with(['items.amount.item.quote.currency'])->find($id);
        event(new MruLog(['module' => 'quotes', 'action' => 'edit', 'id' => $id, 'title' => $quote->number . ' ' . $quote->client->name]));

        $selectedTags = [];

        foreach ($quote->tags as $tagDetail)
        {
            $selectedTags[] = $tagDetail->tag->name;
        }

        return view('quotes.edit')
            ->with('quote', $quote)
            ->with('statuses', QuoteStatuses::lists())
            ->with('currencies', Currency::getList())
            ->with('taxRates', TaxRate::getList())
            ->with('customFields', CustomFieldsParser::getFields('quotes'))
            ->with('quoteItemCustomFields', CustomFieldsParser::getFields('quote_items'))
            ->with('returnUrl', $this->getReturnUrl())
            ->with('tags', Tag::whereTagEntity('sales')->pluck('name', 'name'))
            ->with('selectedTags', $selectedTags)
            ->with('templates', QuoteTemplates::lists())
            ->with('itemCount', count($quote->quoteItems))
            ->with('discountTypes', Quote::getDiscountTypes())
            ->with('currentDate', DateFormatter::format(Carbon::now()))
            ->with('allowLineItemDiscounts', config('fi.allowLineItemDiscounts') == 1 ? true : false)
            ->with('paymentTerms', PaymentTerm::getDropDownList());
    }

    public function oldAndNewObjectCompare($modal)
    {
        $newData = $modal->getAttributes();
        $oldData = $modal->getOriginal();

        unset($newData['created_at'], $oldData['created_at'], $newData['updated_at'], $oldData['updated_at']);
        if (isset($newData['items']))
        {
            unset($newData['items']);
        }
        return !($newData == $oldData);
    }

    public function update(QuoteUpdateRequest $request, $id)
    {
        $quote = Quote::find($id);

        $isChangeQuote     = false;
        $quoteRevisionData = '';
        $documentRevision  = new DocumentRevisionSupport();

        if (count($quote->items) > 0)
        {
            $quoteRevisionData = $documentRevision->prepareArrayForQuote($quote);
        }

        $input               = $request->except(['items', 'custom', 'apply_exchange_rate']);
        $input['quote_date'] = DateFormatter::unformat($input['quote_date']);
        $input['expires_at'] = DateFormatter::unformat($input['expires_at']);

        // Save the quote.
        if (!empty($quote->invoice_id) && in_array(request('status'), ['rejected', 'canceled']))
        {
            if ($quote->invoice->status != 'canceled')
            {
                return response()->json(['errors' => [[trans('fi.quote_rejecting_error', ['invoice_number' => $quote->invoice->number, 'invoice_date' => $quote->invoice->formatted_invoice_date])]]], 400);
            }
        }
        $quote->fill($input);
        $updatedFields = $quote->getDirty();

        $changedQuote = $quote->isDirty() ? $quote->getDirty() : [];
        if (count($changedQuote) > 0)
        {
            $isChangeQuote = $this->oldAndNewObjectCompare($quote);
        }

        if (isset($updatedFields['status']))
        {
            event(new AddTransition($quote, 'status_changed', $quote->getOriginal('status'), $quote->status));

            if ($quote->status == 'approved' && config('fi.convertQuoteWhenApproved') && empty($quote->invoice_id))
            {
                $quoteToInvoice = new QuoteToInvoice();

                $invoice = $quoteToInvoice->convert(
                    $quote,
                    date('Y-m-d'),
                    DateFormatter::incrementDateByDays(date('Y-m-d'), invoicesDueAfter()),
                    config('fi.invoiceGroup')
                );

                event(new QuoteToInvoiceTransition($quote, $invoice));
            }
        }
        $quote->save();

        $newTagsArray = request('tags', []);

        if ($newTagsArray != $quote->tagNames)
        {
            $isChangeQuote = true;
        }

        $manageTags = manageTags($quote, 'quote_tag_updated', 'quote_tag_deleted', 'Quotes');

        $tags = isset($manageTags) ? $manageTags : '';

        $tag_ids = [];

        if (is_array($tags))
        {
            foreach ($tags as $tag)
            {
                $tag = Tag::firstOrNew(['name' => $tag, 'tag_entity' => 'sales'])->fill(['name' => $tag, 'tag_entity' => 'sales']);

                $tag->save();

                $tag_ids[] = $tag->id;
            }
            foreach ($tag_ids as $tag_id)
            {
                $quote->tags()->insert(['quote_id' => $quote->id, 'tag_id' => $tag_id, 'created_at' => Carbon::now(), 'updated_at' => Carbon::now()]);
            }
        }

        $response = '';

        // Save the custom fields.
        $customFieldData = CustomFieldsTransformer::transform(request('custom', []), 'quotes', $quote);
        $quote->custom->update($customFieldData);

        // Save the items.
        foreach ($request->input('items') as $item)
        {
            if (isset($item['discount_type']) && $item['discount_type'] != '')
            {
                if ($item['price'] != 0)
                {
                    if ($item['discount'] != '' && $item['discount_type'] != '')
                    {
                        $priceAndPreviousPrice  = calculateDiscount($item['discount_type'], $item['discount'], $item['price']);
                        $item['previous_price'] = $priceAndPreviousPrice['previous_price'];
                    }
                }
            }

            $item['apply_exchange_rate'] = $request->input('apply_exchange_rate');

            if ($item['name'] == '' and $item['price'] == '')
            {
                continue;
            }

            if (!isset($item['id']) or (!$item['id']))
            {

                $isChangeQuote = true;

                $saveItemAsLookup = $item['save_item_as_lookup'];
                unset($item['save_item_as_lookup']);

                $quoteItem = QuoteItem::create($item);

                if ($item['data_custom_item_delete'] != 'no')
                {
                    if ($item['item_lookup_id'] != '' && isset($item['item_lookup_id']))
                    {
                        $itemCustomImage = findItemLookupsCustomImageField($item['item_lookup_id'], false, 'quote_items');
                        if ($itemCustomImage['customFieldsColumnName'] != '' && $itemCustomImage['itemLookupImageColumn'] != '')
                        {
                            if ($item['custom'][$itemCustomImage['customFieldsColumnName']] == '')
                            {
                                $item['custom'][$itemCustomImage['customFieldsColumnName']] = $itemCustomImage['itemLookUpCustom'][$itemCustomImage['itemLookupImageColumn']];
                            }
                        }
                    }
                }

                if (isset($item['custom']))
                {
                    $customFieldData = CustomFieldsTransformer::transform($item['custom'], 'quote_items', $quoteItem);
                    $quoteItem->custom->update($customFieldData);
                }

                if ($saveItemAsLookup)
                {
                    if (ItemLookup::all()->count() < config('fi.maxItemLookups'))
                    {
                        $itemLookup = ItemLookup::updateOrCreate(['name' => $item['name']], [
                            'name'          => $item['name'],
                            'description'   => $item['description'],
                            'price'         => floatval($item['price']),
                            'quantity'      => abs($item['quantity']),
                            'tax_rate_id'   => isset($item['tax_rate_id']) ? $item['tax_rate_id'] : -1,
                            'tax_rate_2_id' => isset($item['tax_rate_2_id']) ? $item['tax_rate_2_id'] : -1,
                        ]);

                        if (isset($item['custom']))
                        {
                            $customFieldData = CustomFieldsTransformer::sync('quote_items', $item['custom'], 'item_lookups', $itemLookup);
                            $itemLookup->custom->update($customFieldData);
                        }
                    }
                    else
                    {
                        $response = ['error' => trans('fi.item-lookup-overload', ['max_import' => config('fi.maxItemLookupsImport')])];
                    }
                }
            }
            else
            {
                $quoteItem = QuoteItem::find($item['id']);
                $quoteItem->fill($item);

                $isItemChanged = $this->oldAndNewObjectCompare($quoteItem);
                if ($isItemChanged)
                {
                    $isChangeQuote = true;
                }

                $quoteItem->save();


                if ($item['item_lookup_id'] != '' && isset($item['item_lookup_id']))
                {
                    $itemCustomImage = findItemLookupsCustomImageField($item['item_lookup_id'], $item['id'], 'quote_items');
                    if ($item['data_custom_item_delete'] != 'no')
                    {
                        if ($itemCustomImage['customFieldsColumnName'] != '' && $itemCustomImage['itemLookupImageColumn'] != '')
                        {
                            if ($item['custom'][$itemCustomImage['customFieldsColumnName']] == '')
                            {
                                $item['custom'][$itemCustomImage['customFieldsColumnName']] = $itemCustomImage['itemLookUpCustom'][$itemCustomImage['itemLookupImageColumn']];
                            }
                        }
                    }
                    else
                    {
                        removeCustomFieldImage($itemCustomImage['customFieldsColumnName'], 'quote_items', $item['id']);
                    }
                }

                if (isset($item['custom']))
                {
                    $customFieldData = CustomFieldsTransformer::transform($item['custom'], 'quote_items', $quoteItem);
                    $quoteItem->custom->update($customFieldData);
                }

                $saveItemAsLookup = $item['save_item_as_lookup'];

                if ($saveItemAsLookup)
                {
                    if (ItemLookup::all()->count() < config('fi.maxItemLookups'))
                    {
                        $itemLookup = ItemLookup::updateOrCreate(['name' => $item['name']], [
                            'name'          => $item['name'],
                            'description'   => $item['description'],
                            'price'         => floatval($item['price']),
                            'quantity'      => abs($item['quantity']),
                            'tax_rate_id'   => isset($item['tax_rate_id']) ? $item['tax_rate_id'] : -1,
                            'tax_rate_2_id' => isset($item['tax_rate_2_id']) ? $item['tax_rate_2_id'] : -1,
                        ]);
                        if (isset($item['custom']))
                        {
                            $customFieldData = CustomFieldsTransformer::sync('quote_items', $item['custom'], 'item_lookups', $itemLookup);
                            $itemLookup->custom->update($customFieldData);
                        }
                    }
                    else
                    {
                        $response = ['error' => trans('fi.item-lookup-overload', ['max_import' => config('fi.maxItemLookupsImport')])];
                    }
                }
            }
        }
        if (!isset($updatedFields['status']))
        {
            event(new AddTransition($quote, 'updated'));
        }

        if ($isChangeQuote == true && $quoteRevisionData != '')
        {
            $documentRevision->addData($quoteRevisionData['type'], $quoteRevisionData['data']);
        }

        return response()->json($response, 200);
    }

    public function refreshEdit($id)
    {
        $quote        = Quote::with(['items.amount.item.quote.currency'])->find($id);
        $selectedTags = [];
        foreach ($quote->tags as $tagDetail)
        {
            $selectedTags[] = $tagDetail->tag->name;
        }

        return view('quotes._edit')
            ->with('quote', $quote)
            ->with('statuses', QuoteStatuses::lists())
            ->with('currencies', Currency::getList())
            ->with('taxRates', TaxRate::getList())
            ->with('customFields', CustomFieldsParser::getFields('quotes'))
            ->with('quoteItemCustomFields', CustomFieldsParser::getFields('quote_items'))
            ->with('returnUrl', $this->getReturnUrl())
            ->with('templates', QuoteTemplates::lists())
            ->with('tags', Tag::whereTagEntity('sales')->pluck('name', 'name'))
            ->with('selectedTags', $selectedTags)
            ->with('itemCount', count($quote->quoteItems))
            ->with('discountTypes', Quote::getDiscountTypes())
            ->with('currentDate', DateFormatter::format(Carbon::now()))
            ->with('paymentTerms', PaymentTerm::getDropDownList())
            ->with('allowLineItemDiscounts', config('fi.allowLineItemDiscounts') == 1 ? true : false);
    }

    public function refreshTotals()
    {
        return view('quotes._edit_totals')
            ->with('quote', Quote::with(['items.amount.item.quote.currency'])->find(request('id')));
    }

    public function refreshTo()
    {
        return view('quotes._edit_to')
            ->with('quote', Quote::find(request('id')));
    }

    public function refreshFrom()
    {
        return view('quotes._edit_from')
            ->with('quote', Quote::find(request('id')));
    }

    public function updateClient()
    {
        Quote::where('id', request('id'))->update(['client_id' => request('client_id')]);
    }

    public function updateCompanyProfile()
    {
        Quote::where('id', request('id'))->update(['company_profile_id' => request('company_profile_id')]);
    }

    public function deleteImage($id, $columnName)
    {

        if (request('item_custom_id') != 'null' && request('item_custom_id') != '')
        {
            $customFields = QuoteItemCustom::whereQuoteItemId(request('item_custom_id'))->first();
            $existingFile = 'quote_items' . DIRECTORY_SEPARATOR . $customFields->{$columnName};
        }
        else
        {
            $customFields = QuoteCustom::whereQuoteId($id)->first();
            $existingFile = 'quotes' . DIRECTORY_SEPARATOR . $customFields->{$columnName};
        }
        if (Storage::disk(CustomFieldsTransformer::STORAGE_DISK_NAME)->exists($existingFile))
        {
            try
            {
                Storage::disk(CustomFieldsTransformer::STORAGE_DISK_NAME)->delete($existingFile);
                $customFields->{$columnName} = null;
                $customFields->save();
            }
            catch (\Exception $e)
            {
                Log::error($e->getMessage());
            }
        }
    }

    public function addLineItem()
    {

        $quote = Quote::find(request('id'));

        return view('quotes._ajax_add_line_item')
            ->with('quote_id', request('id'))
            ->with('key', request('key'))
            ->with('discountTypes', Quote::getDiscountTypes())
            ->with('currencies', Currency::getList())
            ->with('taxRates', TaxRate::getList())
            ->with('quoteItemCustomFields', CustomFieldsParser::getFields('quote_items'))
            ->with('allowLineItemDiscounts', config('fi.allowLineItemDiscounts') == 1 ? true : false)
            ->with('currencyCode', $quote->currency_code)
            ->render();
    }

    public function updateSummaryAndTags(QuoteUpdateSummaryAndTagRequest $request, $id)
    {
        try
        {
            $quote = Quote::find($id);
            if (count($quote->items) > 0)
            {
                $documentRevision  = new DocumentRevisionSupport();
                $quoteRevisionData = $documentRevision->prepareArrayForQuote($quote);
                if (isset($quoteRevisionData['type']) && $quoteRevisionData['data'] != '')
                {
                    $documentRevision->addData($quoteRevisionData['type'], $quoteRevisionData['data']);
                }
            }
            $quote->summary = $request->summary;
            $quote->save();

            $manageTags = manageTags($quote, 'quote_tag_updated', 'quote_tag_deleted', 'Quotes');

            $tags    = isset($manageTags) ? $manageTags : request('tags', []);
            $tag_ids = [];

            if (is_array($tags))
            {
                foreach ($tags as $tag)
                {
                    $tag = Tag::firstOrNew(['name' => $tag, 'tag_entity' => 'sales'])->fill(['name' => $tag, 'tag_entity' => 'sales']);

                    $tag->save();

                    $tag_ids[] = $tag->id;
                }
                foreach ($tag_ids as $tag_id)
                {
                    $quote->tags()->insert(['quote_id' => $quote->id, 'tag_id' => $tag_id, 'created_at' => Carbon::now(), 'updated_at' => Carbon::now()]);
                }
            }

            return response()->json(['success' => true, 'message' => trans('fi.record_successfully_updated')], 200);
        }
        catch (\Exception $e)
        {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 401);
        }
    }

    public function getPaymentTermDescription($id)
    {
        try
        {
            $quote       = Quote::find($id);
            $paymentTerm = PaymentTerm::find(request('paymentTermId', $quote->paymentTerm->id));

            if ($paymentTerm && $paymentTerm->quote_description_template != '')
            {
                $quotePaymentTerm = new QuotePaymentTerm();

                switch ($paymentTerm->type)
                {
                    case 1:
                        $typeNotice = $quotePaymentTerm->processTypeOne($paymentTerm, $quote);
                        break;
                    case 2:
                        $typeNotice = $quotePaymentTerm->processTypeTwo($paymentTerm, $quote);
                        break;
                    case 3:
                        $typeNotice = $quotePaymentTerm->processTypeThree($paymentTerm, $quote);
                        break;
                    case 4:
                        $typeNotice = $quotePaymentTerm->processTypeFour($paymentTerm, $quote);
                        break;
                    case 5:
                        $typeNotice = $quotePaymentTerm->processTypeFive($paymentTerm, $quote);
                        break;
                }

                return response()->json(['success' => true, 'message' => $typeNotice], 200);
            }

            return response()->json(['success' => true, 'message' => $paymentTerm->name], 200);
        }
        catch (\Exception $e)
        {
            Log::error($e->getMessage());
            return response()->json(['success' => false, 'message' => trans('fi.unknown_error')], 401);
        }
    }
}
