<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Payments\Models;

use Carbon\Carbon;
use FI\Support\CurrencyFormatter;
use FI\Support\DateFormatter;
use FI\Support\FileNames;
use FI\Support\HTML;
use FI\Traits\Sortable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class PaymentInvoice extends Model
{
    use Sortable;

    /**
     * Guarded properties
     * @var array
     */
    protected $guarded = ['id'];

    protected $sortable = ['created_at', 'created_at', 'invoices.invoice_date', 'invoices.number', 'invoices.summary'];

    /*
    |--------------------------------------------------------------------------
    | Relationships
    |--------------------------------------------------------------------------
    */

    public function custom()
    {
        return $this->hasMany('FI\Modules\Payments\Models\Payments');
    }

    public function invoice()
    {
        return $this->belongsTo('FI\Modules\Invoices\Models\Invoice');
    }

    public function installment()
    {
        return $this->belongsTo('FI\Modules\PaymentTerms\Models\PaymentTermInstallment', 'installment_number', 'id');
    }

    public function payment()
    {
        return $this->belongsTo('FI\Modules\Payments\Models\Payment');
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors
    |--------------------------------------------------------------------------
    */
    public function getFormattedInvoiceAmountPaidAttribute()
    {
        return CurrencyFormatter::format($this->attributes['invoice_amount_paid'], $this->invoice->currency);
    }

    public function getFormattedConvenienceChargesAttribute()
    {
        return CurrencyFormatter::format($this->attributes['convenience_charges'], $this->invoice->currency);
    }

    public function getFormattedPaymentTermsDiscountAttribute()
    {
        return CurrencyFormatter::format($this->attributes['payment_terms_discount'], $this->invoice->currency);
    }

    public function getFormattedCreatedAtAttribute()
    {
        return DateFormatter::format($this->attributes['created_at']);
    }

    public function getFormattedPaidAtAttribute()
    {
        return DateFormatter::format($this->payment->paid_at);
    }

    public function getFormattedPreviousAmountAttribute()
    {
        $thisPayment      = $this->attributes['invoice_amount_paid'];
        $invoiceTotalPaid = $this->invoice->amount->paid - $thisPayment;
        return ($invoiceTotalPaid > 0) ? CurrencyFormatter::format($invoiceTotalPaid, $this->invoice->currency) : false;
    }

    public function getFormattedPreviousBalanceAttribute()
    {
        $paymentInvoice  = $this;
        $paymentIdsArray = $paymentInvoice->invoice->payments()->orderBy('payment_invoices.id', 'asc')->pluck('id')->toArray() ?? [];
        $position        = array_search($paymentInvoice->id, $paymentIdsArray);

        if ($position)
        {
            $threshold           = $paymentInvoice->id;
            $result              = array_filter($paymentIdsArray, function ($value) use ($threshold)
            {
                return $value < $threshold;
            });
            $preTotalPaidBalance = PaymentInvoice::whereIn('id', $result)->sum(DB::raw('invoice_amount_paid + payment_terms_discount'));
            $total               = floatval($paymentInvoice->invoice->amount->total) + floatval($paymentInvoice->invoice->amount->payment_terms_discount);
            return ($total - floatval($preTotalPaidBalance));
        }

        return '';
    }

    public function getFormattedPreviousBalanceWithCurrencyAttribute()
    {
        if ($this->formatted_previous_balance)
        {
            return CurrencyFormatter::format($this->formatted_previous_balance, $this->invoice->currency);
        }
        return '';
    }

    public function getFormattedPreviousRemainingBalanceAttribute()
    {
        $paymentInvoice      = $this;
        $paymentIdsArray     = $paymentInvoice->invoice->payments()->orderBy('payment_invoices.id', 'asc')->pluck('id')->toArray() ?? [];
        $paymentInvoiceId    = $paymentInvoice->id;
        $result              = array_filter($paymentIdsArray, function ($value) use ($paymentInvoiceId)
        {
            return $value <= $paymentInvoiceId;
        });
        $preTotalPaidBalance = PaymentInvoice::whereIn('id', $result)->sum(DB::raw('invoice_amount_paid + payment_terms_discount')) ?? 0;
        $total               = floatval($this->invoice->amount->total) + floatval($this->invoice->amount->payment_terms_discount);

        $previousRemainingBalance = $total - floatval($preTotalPaidBalance);
        return CurrencyFormatter::format($previousRemainingBalance, $this->invoice->currency);
    }


    public function getFormattedInvoiceTotalForPdfAttribute()
    {
        $total = floatval($this->invoice->amount->total) + floatval($this->invoice->amount->payment_terms_discount);
        return CurrencyFormatter::format($total, $this->invoice->currency);
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeThisYear($query)
    {
        return $query->where(DB::raw('YEAR(' . DB::getTablePrefix() . 'payment_invoices.created_at)'), '=', DB::raw('YEAR(CURRENT_DATE())'));
    }

    public function scopeThisMonth($query)
    {
        return $query->where(DB::raw('MONTH(' . DB::getTablePrefix() . 'payment_invoices.created_at)'), '=', DB::raw('MONTH(CURRENT_DATE())'))
            ->where(DB::raw('YEAR(' . DB::getTablePrefix() . 'payment_invoices.created_at)'), '=', DB::raw('YEAR(CURRENT_DATE())'));
    }

    public function scopeThisQuarter($query)
    {
        return $query->where('payment_invoices.created_at', '>=', Carbon::now()->firstOfQuarter())
            ->where('payment_invoices.created_at', '<=', Carbon::now()->lastOfQuarter());
    }

    public function scopeLastMonth($query)
    {
        return $query->where(DB::raw('payment_invoices.created_at'), '>=', Carbon::now()->subMonths(1)->firstOfMonth())
            ->where(DB::raw('payment_invoices.created_at'), '<=', Carbon::now()->subMonths(1)->lastOfMonth());
    }

// -mw-

    public function scopeLastQuarter($query)
    {
        return $query->where('payment_invoices.created_at', '>=', Carbon::now()->subQuarters(1)->firstOfQuarter())
            ->where('payment_invoices.created_at', '<=', Carbon::now()->subQuarters(1)->lastOfQuarter());
    }

    public function scopeLastYear($query)
    {
        return $query->where(DB::getTablePrefix() . 'payment_invoices.created_at', '>=', Carbon::now()->subYears(1)->firstOfYear())
            ->where(DB::getTablePrefix() . 'payment_invoices.created_at', '<=', Carbon::now()->subYears(1)->lastOfYear());
    }

    public function scopeDateRange($query, $from, $to)
    {
        return $query->where('payment_invoices.created_at', '>=', Carbon::parse($from)->format('Y-m-d'))->where('payment_invoices.created_at', '<=', Carbon::parse($to)->format('Y-m-d'));
    }

    public function scopeYear($query, $year)
    {
        return $query->where(DB::raw('YEAR(' . DB::getTablePrefix() . 'payment_invoices.created_at)'), '=', $year);
    }

    public function scopeClientId($query, $clientId)
    {
        if ($clientId)
        {
            $query->whereHas('invoice', function ($query) use ($clientId)
            {
                $query->where('client_id', $clientId);
            });
        }

        return $query;
    }

    public function getFormattedAmountAttribute()
    {
        return CurrencyFormatter::format($this->attributes['amount'], $this->invoice->currency);
    }

    public function getUserAttribute()
    {
        return $this->invoice->user;
    }

    public function getHtmlAttribute()
    {
        return HTML::invoice($this->invoice);
    }

    public function getPdfFilenameAttribute()
    {
        return FileNames::invoice($this->invoice);
    }
}