<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Payments\Controllers;

use FI\Http\Controllers\Controller;
use FI\Modules\MailQueue\Support\MailQueue;
use FI\Modules\Payments\Events\AddTransition;
use FI\Modules\Payments\Models\Payment;
use FI\Requests\SendEmailRequest;
use FI\Support\Contacts;
use FI\Support\Parser;

class PaymentMailController extends Controller
{
    private $mailQueue;

    public function __construct(MailQueue $mailQueue)
    {
        $this->mailQueue = $mailQueue;
    }

    public function create()
    {
        $payment = Payment::find(request('payment_id'));

        $contacts = new Contacts($payment->client);

        $parser = new Parser($payment);

        if (strtolower($payment->user->name) === 'system')
        {
            $fromMail = [
                config('fi.mailFromName') . '###' . config('fi.mailFromAddress') => config('fi.mailFromAddress'),
                auth()->user()->name . '###' . auth()->user()->email             => auth()->user()->email,
            ];
        }
        else
        {
            $fromMail = [
                config('fi.mailFromName') . '###' . config('fi.mailFromAddress') => config('fi.mailFromAddress'),
                $payment->user->name . '###' . $payment->user->email             => $payment->user->email,
                auth()->user()->name . '###' . auth()->user()->email             => auth()->user()->email,
            ];
        }

        return view('payments._modal_mail')
            ->with('paymentId', $payment->id)
            ->with('redirectTo', request('redirectTo'))
            ->with('subject', $parser->companyProfileMailSubjectParse('paymentReceipt', $payment->client->companyProfile->payment_receipt_email_body))
            ->with('body', $parser->companyProfileMailBodyParse('paymentReceipt', $payment->client->companyProfile->payment_receipt_email_body))
            ->with('contactDropdownTo', $contacts->contactDropdownTo(false,3))
            ->with('contactDropdownCc', $contacts->contactDropdownCc(3))
            ->with('contactDropdownBcc', $contacts->contactDropdownBcc(3))
            ->with('fromMail', $fromMail);
    }

    public function store(SendEmailRequest $request)
    {
        if (!config('app.demo'))
        {
            $payment = Payment::find($request->input('payment_id'));

            $input                   = $request->except('payment_id');
            $input['attach_pdf']     = 0;
            $input['attach_invoice'] = $request->get('attach_pdf', 0);
            $mail                    = $this->mailQueue->create($payment, $input);

            event(new AddTransition($payment, 'payment_receipt_email_sent'));

            if (!$this->mailQueue->send($mail->id))
            {
                return response()->json(['errors' => [[$this->mailQueue->getError()]]], 400);
            }
        }
        else
        {
            return response()->json(['errors' => [[trans('fi.functionality_not_available_on_demo')]]], 400);
        }
    }
}
