<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\PaymentTerms\Rule;

use FI\Modules\PaymentTerms\Models\PaymentTerm;
use Illuminate\Contracts\Validation\Rule;

class PaymentTermRule implements Rule
{

    public array $errors;
    public array $inputs;
    public       $type;
    private      $id;

    public function __construct()
    {
        $this->errors = [];
        $this->type   = request('type');
        $this->inputs = request()->except('name', 'type', '_token');
        $this->id     = request('id');
    }

    public function passes($attribute, $value)
    {
        $modifiedArray = self::filterData($this->inputs, $this->type);

        foreach ($modifiedArray as $key => $value)
        {
            if (empty($value) || ($value < 0))
            {
                if ($key == 'discount_percent')
                {
                    $this->errors[] = trans('fi.payment_terms_error.discount_percent_gt_error');
                }
                else if ($key == 'down_payment')
                {
                    $this->errors[] = trans('fi.payment_terms_error.down_payment_gt_error');
                }
                else if (substr($key, 0, -2) === 'installment_percent' || substr($key, 0, -2) === 'installment_due_days')
                {
                    if (substr($key, -1, 1) != 1)
                    {
                        if (substr($key, 0, -2) === 'installment_percent')
                        {
                            $this->errors[] = trans('fi.payment_terms_error.installment_percent_error', ['index' => substr($key, -1)]);
                        }
                        if (substr($key, 0, -2) === 'installment_due_days')
                        {
                            $this->errors[] = trans('fi.payment_terms_error.installment_due_days_error', ['index' => substr($key, -1)]);
                        }
                    }
                }
                else
                {
                    if ($value != 0 && $key != 'net_due_days')
                    {
                        $this->errors[] = trans('fi.payment_terms_error.' . $key . '_error');
                    }
                    if (count($modifiedArray) == 1 && $key == 'net_due_days')
                    {
                        $this->errors[] = trans('fi.payment_terms_error.' . $key . '_error');
                    }
                }
            }
            else
            {

                if ($key == 'discount_percent')
                {
                    if (!is_numeric($value))
                    {
                        $this->errors[] = trans('fi.payment_terms_error.discount_percent_numeric_error');
                    }
                    if (is_numeric($value) && $value > 100)
                    {
                        $this->errors[] = trans('fi.payment_terms_error.discount_percent_lt_error');
                    }
                }
            }
        }

        if ($this->type == 2 || $this->type == 3)
        {
            if (isset($modifiedArray['discount_days_simple']) && $modifiedArray['discount_days_simple'] > $modifiedArray['net_due_days'])
            {
                $this->errors[] = trans('fi.payment_terms_error.discount_days_gt_net_days_error');
            }
            if (isset($modifiedArray['discount_days_nth']) && $modifiedArray['discount_days_nth'] > $modifiedArray['net_due_days'])
            {
                $this->errors[] = trans('fi.payment_terms_error.discount_days_nth_gt_net_days_error');
            }
        }

        if ($this->type == 4)
        {
            if (request('down_payment_type') == 'fa-percentage')
            {
                if (floatval($modifiedArray['down_payment']) > 100)
                {
                    $this->errors[] = trans('fi.payment_terms_error.down_payment_percent_lt_error');
                }
                if (preg_match('/#DOWNPAYMENT(?!PERCENT)/', request('description_template')))
                {
                    $this->errors[] = trans('fi.payment_terms_error.down_payment_type_invoice_description_error');
                }
                if (preg_match('/#DOWNPAYMENT(?!PERCENT)/', request('quote_description_template')))
                {
                    $this->errors[] = trans('fi.payment_terms_error.down_payment_type_quote_description_error');
                }
            }
            if (request('down_payment_type') == 'fa-money-bill')
            {
                if (preg_match('/#DOWNPAYMENTPERCENT/', request('description_template')))
                {
                    $this->errors[] = trans('fi.payment_terms_error.down_payment_type_invoice_description_error');
                }
                if (preg_match('/#DOWNPAYMENTPERCENT/', request('quote_description_template')))
                {
                    $this->errors[] = trans('fi.payment_terms_error.down_payment_type_quote_description_error');
                }
            }
        }

        if ($this->type == 5)
        {
            $percentage = 0;
            $numeric    = false;


            for ($i = 1; $i <= $modifiedArray['number_of_installments']; $i++)
            {

                if (intval($modifiedArray['installment_due_days_' . $i]) < 0)
                {
                    $this->errors[] = trans('fi.payment_terms_error.installment_due_days_gt_zero_error', ['index' => $i]);
                }

                if ($i >= 2)
                {
                    $percentage += floatval($modifiedArray['installment_percent_' . $i]);
                    $preIndex   = $i - 1;
                    if ($modifiedArray['installment_due_days_' . $preIndex] > $modifiedArray['installment_due_days_' . $i])
                    {
                        $this->errors[] = trans('fi.payment_terms_error.installment_due_days_gt_error', ['preIndex' => $preIndex, 'index' => $i]);
                    }
                }
                else
                {
                    if (!is_numeric($modifiedArray['installment_percent_' . $i]))
                    {
                        $this->errors[] = trans('fi.payment_terms_error.installment_percent_numeric_error');
                    }
                    else
                    {
                        $percentage = floatval($modifiedArray['installment_percent_' . $i]);

                        $numeric = true;
                    }
                }
            }

            if ($numeric && 100 != intval(round($percentage)))
            {
                $this->errors[] = trans('fi.payment_terms_error.installment_percent_total_error');
            }

        }

        if (!empty($this->id))
        {
            $paymentTerm = PaymentTerm::find($this->id);

            if (!empty($paymentTerm) && count($paymentTerm->invoices) > 0)
            {
                if ($paymentTerm->status != $this->inputs['status'])
                {
                    $this->errors[] = trans('fi.payment_terms_error.status_error');
                }
            }
        }

        if (count($this->errors) > 0)
        {
            return false;
        }
        else
        {
            return true;
        }
    }

    public static function filterData($inputs, $type)
    {

        $validationArray = array_filter($inputs, function ($key) use ($type)
        {
            if ($type != 5)
            {
                return substr($key, -2) === '_' . $type;
            }
            else
            {
                return substr($key, -2) === '_' . $type || substr($key, 0, -2) === 'installment_percent' || substr($key, 0, -2) === 'installment_due_days';
            }
        }, ARRAY_FILTER_USE_KEY);

        $modifiedArray = [];

        foreach ($validationArray as $key => $value)
        {
            $newKey = substr($key, 0, -2);
            if ($newKey === 'installment_percent' || $newKey === 'installment_due_days')
            {
                $modifiedArray[$key] = $value;
            }
            else
            {
                $modifiedArray[$newKey] = $value;
            }
        }

        return $modifiedArray;
    }

    public function message()
    {
        return $this->errors;
    }
}