<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\PaymentTerms\Controllers;

use FI\Http\Controllers\Controller;
use FI\Modules\PaymentTerms\Models\PaymentTerm;
use FI\Modules\PaymentTerms\Models\PaymentTermInstallment;
use FI\Modules\PaymentTerms\Requests\PaymentTermEditableRequest;
use FI\Modules\PaymentTerms\Requests\PaymentTermRequest;
use Illuminate\Support\Facades\Log;

class PaymentTermController extends Controller
{
    public function index()
    {
        try
        {
            $paymentTerms = PaymentTerm::sortable(['name' => 'asc', 'status' => 'asc'])
                ->keywords(request('search'))
                ->paginate(config('fi.resultsPerPage'));

            $defaultPayment = PaymentTerm::whereIsDefault(1)->whereType(0)->first();

            return view('payment_terms.index')->with('searchPlaceholder', trans('fi.search_items'))
                ->with('paymentTerms', $paymentTerms)
                ->with('defaultPaymentId', $defaultPayment->id);
        }
        catch (\Exception $e)
        {
            Log::error($e->getMessage());
        }

    }

    public function create()
    {
        return view('payment_terms.form')
            ->with('status', [1 => trans('fi.active'), 0 => trans('fi.inactive')])
            ->with('redirect', route('payment.terms.index'))
            ->with('numberOfInstallments', PaymentTerm::numberOfInstallments())
            ->with('editMode', false);
    }

    public function store(PaymentTermRequest $request)
    {
        try
        {
            $type = $request->type;

            $number_of_installments = $request->get("number_of_installments_" . $type);

            $payment_term_data = [
                "type"                       => $type,
                "name"                       => $request->name,
                "net_due_days"               => $request->get("net_due_days_" . $type),
                "discount_percent"           => $request->get("discount_percent_" . $type),
                "discount_days_simple"       => $request->get("discount_days_simple_" . $type),
                "discount_days_nth"          => $request->get("discount_days_nth_" . $type),
                "down_payment"               => $request->get("down_payment_" . $type),
                "balance_due_days"           => $request->get("balance_due_days_" . $type),
                "number_of_installments"     => $number_of_installments,
                'description_template'       => $request->description_template,
                'quote_description_template' => $request->quote_description_template,
                'status'                     => $request->status,
                'down_payment_type'          => $request->down_payment_type,
            ];

            $paymentTerm = PaymentTerm::create($payment_term_data);

            if ($type == 5)
            {
                for ($i = 1; $i <= $number_of_installments; $i++)
                {
                    $paymentTermInstallment = [
                        "installment_percent"  => request("installment_percent_" . $i),
                        "installment_due_days" => request("installment_due_days_" . $i),
                        "installment_number"   => $i,
                        "payment_terms_id"     => $paymentTerm->id,
                    ];

                    PaymentTermInstallment::create($paymentTermInstallment);
                }
            }

            return response()->json(['success' => true, 'redirect' => route('payment.terms.index'), 'message' => trans('fi.record_successfully_updated')], 200);
        }
        catch (\Exception $e)
        {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 400);
        }
    }

    public function edit($id)
    {
        $paymentTerm = PaymentTerm::with('installments')->find($id);

        $invoiceStatus = count($paymentTerm->invoices) > 0 ? $paymentTerm->invoices->pluck('status')->toArray() : [];

        $defaultPayment = PaymentTerm::whereIsDefault(1)->first();

        $notEditable = in_array('sent', $invoiceStatus) || in_array('applied', $invoiceStatus);

        $typeZero = $defaultPayment->id == $paymentTerm->id;

        return view('payment_terms.form')->with('paymentTerm', $paymentTerm)
            ->with('redirect', route('payment.terms.index'))
            ->with('status', [1 => trans('fi.active'), 0 => trans('fi.inactive')])
            ->with('numberOfInstallments', PaymentTerm::numberOfInstallments())
            ->with('notEditable', $notEditable)
            ->with('defaultPaymentId', $defaultPayment->id)
            ->with('typeZero', $typeZero)
            ->with('editMode', true);
    }

    public function update(PaymentTermRequest $request, $id)
    {
        try
        {
            $paymentTerm            = PaymentTerm::find($id);
            $type                   = request('type');
            $number_of_installments = request("number_of_installments_" . $type);

            $payment_term_data = [
                "type"                       => $type,
                "name"                       => request("name"),
                "net_due_days"               => request("net_due_days_" . $type),
                "discount_percent"           => request("discount_percent_" . $type),
                "discount_days_simple"       => request("discount_days_simple_" . $type),
                "discount_days_nth"          => request("discount_days_nth_" . $type),
                "down_payment"               => request("down_payment_" . $type),
                "balance_due_days"           => request("balance_due_days_" . $type),
                "number_of_installments"     => $number_of_installments,
                'description_template'       => $request->description_template,
                'quote_description_template' => $request->quote_description_template,
                'status'                     => $request->status,
                'down_payment_type'          => $request->down_payment_type,
            ];

            $paymentTerm->update($payment_term_data);

            if ($type == 5)
            {
                PaymentTermInstallment::where('payment_terms_id', $id)->delete();
                for ($i = 1; $i <= $number_of_installments; $i++)
                {
                    PaymentTermInstallment::updateOrCreate(
                        ['installment_number' => $i, 'payment_terms_id' => $id],
                        ["installment_percent"  => request("installment_percent_" . $i),
                         "installment_due_days" => request("installment_due_days_" . $i),
                         "installment_number"   => $i,
                         "payment_terms_id"     => $id]
                    );
                }
            }
            else
            {
                PaymentTermInstallment::where('payment_terms_id', $id)->delete();
            }

            return response()->json(['success' => true, 'redirect' => route('payment.terms.index'), 'message' => trans('fi.record_successfully_updated')], 200);
        }
        catch (\Exception $e)
        {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 400);
        }
    }

    public function delete($id)
    {
        try
        {
            $paymentTerm = PaymentTerm::whereId($id)->first();
            $modal       = [];

            if (empty($paymentTerm->invoices) != true && count($paymentTerm->invoices) > 0)
            {
                $modal[] = trans('fi.invoice');
            }
            if (empty($paymentTerm->quotes) != true && count($paymentTerm->quotes) > 0)
            {
                $modal[] = trans('fi.quote');
            }
            if ($paymentTerm->companyProfiles->isNotEmpty())
            {
                $modal[] = trans('fi.company_profile');
            }

            if (count($modal) > 0)
            {
                return response()->json(['success' => false, 'message' => trans('fi.payment_terms_applied', ['modal' => implode(', ', $modal)])], 400);
            }

            PaymentTerm::destroy($id);

            return response()->json(['success' => true, 'message' => trans('fi.record_successfully_deleted')], 200);
        }
        catch (\Exception $e)
        {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 400);
        }
    }

    public function deleteModal()
    {
        try
        {
            return view('layouts._delete_modal')->with('url', request('action'))->with('modalName', request('modalName'))->with('isReload', request('isReload'));
        }
        catch (\Exception $e)
        {
            return response()->json(['success' => false, 'message' => ($e->getCode() == 0) ? trans('fi.modal_not_found') : $e->getMessage()], 400);
        }
    }

    public function editableUpdate(PaymentTermEditableRequest $request, $id)
    {
        $paymentTerm                             = PaymentTerm::find($id);
        $paymentTerm->name                       = $request->name;
        $paymentTerm->description_template       = $request->description_template;
        $paymentTerm->quote_description_template = $request->quote_description_template;
        $paymentTerm->save();

        return response()->json(['success' => true, 'redirect' => route('payment.terms.index'), 'message' => trans('fi.record_successfully_updated')], 200);
    }
}
