<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Merchant\Support\Drivers;

use Carbon\Carbon;
use FI\Modules\Invoices\Models\Invoice;
use FI\Modules\Merchant\Models\MerchantPayment;
use FI\Modules\Merchant\Support\MerchantDriver;
use FI\Modules\PaymentMethods\Models\PaymentMethod;
use FI\Modules\Payments\Events\PaymentEmailed;
use FI\Modules\Payments\Models\Payment;
use FI\Modules\Payments\Models\PaymentInvoice;
use FI\Modules\Quotes\Events\QuoteApproved;
use FI\Modules\Quotes\Models\Quote;
use FI\Modules\Users\Models\User;
use Illuminate\Support\Facades\Log;
use Stripe\Checkout\Session as StripeSession;
use Stripe\PaymentIntent;
use Stripe\Stripe;

class StripeDriver extends MerchantDriver
{
    public function getSettings()
    {
        return ['publishableKey', 'secretKey', 'apiVersion' => ['2021' => trans('fi.2021_prior'), '2022' => trans('fi.2022_later')]];
    }

    public function pay(Invoice $invoice, $modal, $quoteUrlKey)
    {
        $onlinePaymentChargesDetail = onlinePaymentChargesDetail($invoice);
        $amountValue                = $onlinePaymentChargesDetail['amountValue'];

        Stripe::setApiKey($this->getSetting('secretKey'));

        try
        {
            if ($this->getSetting('apiVersion') == 2022)
            {
                $session = StripeSession::create([
                    'line_items'  => [
                        [
                            'price_data' =>
                                [
                                    'currency'     => $invoice->currency_code,
                                    'unit_amount'  => $amountValue * 100,
                                    'product_data' => [
                                        'name'        => trans('invoice') . ' #' . $invoice->number,
                                        'description' => trans('invoice') . ' #' . $invoice->number,
                                    ],
                                ],
                            'quantity'   => 1,
                        ],

                    ],
                    'mode'        => 'payment',
                    'success_url' => route('merchant.pay.stripe.return', [$invoice->url_key, $modal, $quoteUrlKey]) . '?session_id={CHECKOUT_SESSION_ID}',
                    'cancel_url'  => route('merchant.pay.stripe.cancel', [$invoice->url_key, $modal, $quoteUrlKey]),
                ]);
                return view('merchant.stripe')
                    ->with('stripeSessionId', $session->id);
            }
            else
            {
                Stripe::setApiVersion('2020-08-27');

                $session = StripeSession::create([
                    'line_items'  => [
                        [
                            'name'        => trans('invoice') . ' #' . $invoice->number,
                            'description' => trans('invoice') . ' #' . $invoice->number,
                            'amount'      => $amountValue * 100,
                            'currency'    => $invoice->currency_code,
                            'quantity'    => 1,
                        ],
                    ],
                    'success_url' => route('merchant.pay.stripe.return', [$invoice->url_key, $modal, $quoteUrlKey]) . '?session_id={CHECKOUT_SESSION_ID}',
                    'cancel_url'  => route('merchant.pay.stripe.cancel', [$invoice->url_key, $modal, $quoteUrlKey]),
                ]);

                return view('merchant.stripe')
                    ->with('stripeSessionId', $session->id);
            }
        }
        catch (\Exception $e)
        {
            Log::error($e->getMessage());

            if ($quoteUrlKey != '')
            {
                if ($modal == 'pi' && class_exists(\Addons\ProformaInvoice\Models\ProformaInvoice::class) && config('proforma_invoice_enabled') == true)
                {
                    $proformaInvoice = \Addons\ProformaInvoice\Models\ProformaInvoice::where('url_key', $quoteUrlKey)->first();
                    $proformaInvoice->invoice->delete();
                    return redirect()->route('clientCenter.public.proformaInvoice.show', [$quoteUrlKey, $proformaInvoice->token])
                        ->with('alert', trans('fi.payment_problem'));
                }
                else
                {
                    $quote = Quote::where('url_key', $quoteUrlKey)->first();
                    $quote->invoice->delete();
                    return redirect()->route('clientCenter.public.quote.show', [$quoteUrlKey, $quote->token])
                        ->with('alert', trans('fi.payment_problem'));
                }
            }

            return redirect()->route('clientCenter.public.invoice.show', [$invoice->url_key, $invoice->token])
                ->with('error', trans('fi.payment_problem'));

        }
    }

    public function success(Invoice $invoice, $modal, $quoteUrlKey)
    {
        $onlinePaymentChargesDetail = onlinePaymentChargesDetail($invoice);
        $paymentTermsDiscount       = $onlinePaymentChargesDetail['paymentTermsDiscount'];

        $stripeConvenienceCharges = 0;

        if ($invoice->online_payment_processing_fee == 'yes' && config('fi.feePercentage') != '')
        {
            $stripeConvenienceCharges = $onlinePaymentChargesDetail['feeCharges'];
        }

        Stripe::setApiKey($this->getSetting('secretKey'));

        $session = StripeSession::retrieve(request('session_id'));

        $paymentIntent = PaymentIntent::retrieve($session->payment_intent);

        if ($paymentIntent->status == 'succeeded')
        {
            onlineConvenienceCharges($invoice, $onlinePaymentChargesDetail['feeCharges']);

            if ($paymentTermsDiscount != 0)
            {
                $invoice->amount->payment_terms_discount += $paymentTermsDiscount;
                $invoice->amount->save();
            }

            $userId              = (auth()->user() != null) ? ((auth()->user()->user_type == 'paymentcenter_user') ? auth()->user()->id : User::whereUserType('system')->first()->id) : User::whereUserType('system')->first()->id;
            $onlinePaymentMethod = PaymentMethod::whereName(trans('fi.online_payment'))->first();
            $paymentIntentAmount = ($paymentIntent->amount / 100);
            $fiPayment           = Payment::create([
                'user_id'           => $userId,
                'client_id'         => $invoice->client->id,
                'amount'            => $paymentIntentAmount,
                'currency_code'     => $invoice->currency_code,
                'remaining_balance' => 0,
                'payment_method_id' => ($onlinePaymentMethod != null) ? $onlinePaymentMethod->id : config('fi.onlinePaymentMethod'),
                'paid_at'           => Carbon::now()->format('Y-m-d'),
            ]);

            if ($fiPayment)
            {
                $paymentInvoice                         = new PaymentInvoice();
                $paymentInvoice->payment_id             = $fiPayment->id;
                $paymentInvoice->invoice_id             = $invoice->id;
                $paymentInvoice->invoice_amount_paid    = $paymentIntentAmount;
                $paymentInvoice->convenience_charges    = $stripeConvenienceCharges;
                $paymentInvoice->payment_terms_discount = $paymentTermsDiscount;
                $paymentInvoice->installment_number     = $onlinePaymentChargesDetail['installment'];
                $paymentInvoice->save();

                if (!empty($quoteUrlKey) && $quoteUrlKey != 'fi')
                {
                    $quote = Quote::whereUrlKey($quoteUrlKey)->first();
                    if (isset($quote->id))
                    {
                        event(new QuoteApproved($quote, 'quote_payment_successfully'));
                    }
                }

                if ($fiPayment->client->should_email_payment_receipt)
                {
                    event(new PaymentEmailed($fiPayment));
                }
            }
            if (isset($paymentIntent->id) && isset($fiPayment->id))
            {
                MerchantPayment::saveByKey('Stripe', $fiPayment->id, 'id', $paymentIntent->id);
            }
            return redirect()->route('clientCenter.public.invoice.show', [$invoice->url_key, $invoice->token])
                ->with('alertSuccess', trans('fi.payment_applied'));
        }

        if ($quoteUrlKey != '')
        {
            if ($modal == 'pi' && class_exists(\Addons\ProformaInvoice\Models\ProformaInvoice::class) && config('proforma_invoice_enabled') == true)
            {
                $proformaInvoice = \Addons\ProformaInvoice\Models\ProformaInvoice::where('url_key', $quoteUrlKey)->first();
                $proformaInvoice->invoice->delete();
                return redirect()->route('clientCenter.public.proformaInvoice.show', [$quoteUrlKey, $proformaInvoice->token])
                    ->with('alert', trans('fi.payment_problem'));
            }
            else
            {
                $quote = Quote::where('url_key', $quoteUrlKey)->first();
                $quote->invoice->delete();
                return redirect()->route('clientCenter.public.quote.show', [$quoteUrlKey, $quote->token])
                    ->with('alert', trans('fi.payment_problem'));
            }

        }

        return redirect()->route('clientCenter.public.invoice.show', [$invoice->url_key, $invoice->token])
            ->with('alert', trans('fi.payment_problem'));
    }

    public function cancel($urlKey, $token = null, $modal = 'invoice', $quoteUrlKey = null)
    {
        if ($quoteUrlKey != '')
        {
            if ($modal == 'pi' && class_exists(\Addons\ProformaInvoice\Models\ProformaInvoice::class) && config('proforma_invoice_enabled') == true)
            {
                $proformaInvoice = \Addons\ProformaInvoice\Models\ProformaInvoice::where('url_key', $quoteUrlKey)->first();
                $proformaInvoice->invoice->delete();
                return redirect()->route('clientCenter.public.proformaInvoice.show', [$quoteUrlKey, $proformaInvoice->token])
                    ->with('alert', trans('fi.payment_canceled'));
            }
            else
            {
                $quote = Quote::where('url_key', $quoteUrlKey)->first();
                $quote->invoice->delete();
                return redirect()->route('clientCenter.public.quote.show', [$quoteUrlKey, $quote->token])
                    ->with('alert', trans('fi.payment_canceled'));
            }

        }

        return redirect()->route('clientCenter.public.invoice.show', [$urlKey, $token])
            ->with('alert', trans('fi.payment_canceled'));
    }
}