<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Merchant\Support\Drivers;

use Carbon\Carbon;
use FI\Modules\Invoices\Models\Invoice;
use FI\Modules\Merchant\Models\MerchantPayment;
use FI\Modules\Merchant\Support\MerchantDriver;
use FI\Modules\PaymentMethods\Models\PaymentMethod;
use FI\Modules\Payments\Events\PaymentEmailed;
use FI\Modules\Payments\Models\Payment;
use FI\Modules\Payments\Models\PaymentInvoice;
use FI\Modules\Quotes\Events\QuoteApproved;
use FI\Modules\Quotes\Models\Quote;
use FI\Modules\Users\Models\User;
use Illuminate\Support\Facades\Log;
use Mollie\Api\MollieApiClient;

class MollieDriver extends MerchantDriver
{
    public function getSettings()
    {
        return ['apiKey'];
    }

    public function pay(Invoice $invoice, $modal, $quoteUrlKey)
    {
        try
        {
            $onlinePaymentChargesDetail = onlinePaymentChargesDetail($invoice);

            $amountValue = number_format($onlinePaymentChargesDetail['amountValue'], 2, '.', '');

            $mollie = new MollieApiClient();

            $mollie->setApiKey($this->getSetting('apiKey'));

            $payment = $mollie->payments->create(
                [
                    'amount'      =>
                        [
                            'currency' => $invoice->currency_code,
                            'value'    => $amountValue,
                        ],
                    'description' => trans('fi.invoice') . ' #' . $invoice->number,
                    'redirectUrl' => route('merchant.pay.mollie.return', [$invoice->url_key, $modal, $quoteUrlKey]),
                    'webhookUrl'  => route('merchant.pay.mollie.webhook', [$invoice->url_key, $modal, $quoteUrlKey]),
                ]
            );

            return $payment->getCheckoutUrl();
        }
        catch (\Exception $e)
        {
            Log::error($e->getMessage());

            if ($quoteUrlKey != '')
            {
                if ($modal == 'pi' && class_exists(\Addons\ProformaInvoice\Models\ProformaInvoice::class) && config('proforma_invoice_enabled') == true)
                {
                    $proformaInvoice = \Addons\ProformaInvoice\Models\ProformaInvoice::where('url_key', $quoteUrlKey)->first();
                    $proformaInvoice->invoice->delete();
                    return redirect()->route('clientCenter.public.proformaInvoice.show', [$quoteUrlKey, $proformaInvoice->token])
                        ->with('alert', trans('fi.payment_problem'));
                }
                else
                {
                    $quote = Quote::where('url_key', $quoteUrlKey)->first();
                    $quote->invoice->delete();
                    return redirect()->route('clientCenter.public.quote.show', [$quoteUrlKey, $quote->token])
                        ->with('alert', trans('fi.payment_problem'));
                }

            }

            return redirect()->route('clientCenter.public.invoice.show', [$invoice->url_key, $invoice->token])
                ->with('error', trans('fi.payment_problem'));

        }
    }

    public function verify(Invoice $invoice, $modal, $quoteUrlKey)
    {
        try
        {

            $onlinePaymentChargesDetail = onlinePaymentChargesDetail($invoice);
            $paymentTermsDiscount       = $onlinePaymentChargesDetail['paymentTermsDiscount'];

            $mollieConvenienceCharges = 0;

            if ($invoice->online_payment_processing_fee == 'yes' && config('fi.feePercentage') != '')
            {
                $mollieConvenienceCharges = $onlinePaymentChargesDetail['feeCharges'];
            }

            $mollie = new MollieApiClient();

            $mollie->setApiKey($this->getSetting('apiKey'));

            $payment = $mollie->payments->get(request('id'));

            if ($payment->isPaid() && (($invoice->amount->balance == 0 && $invoice->amount->total > 0 && $invoice->status != 'canceled') != true))
            {
                onlineConvenienceCharges($invoice, $onlinePaymentChargesDetail['feeCharges']);
                if ($paymentTermsDiscount != 0)
                {
                    $invoice->amount->payment_terms_discount += $paymentTermsDiscount;
                    $invoice->amount->save();
                }

                $userId              = (auth()->user() != null) ? ((auth()->user()->user_type == 'paymentcenter_user') ? auth()->user()->id : User::whereUserType('system')->first()->id) : User::whereUserType('system')->first()->id;
                $onlinePaymentMethod = PaymentMethod::whereName(trans('fi.online_payment'))->first();

                $fiPayment = Payment::create([
                    'user_id'           => $userId,
                    'client_id'         => $invoice->client->id,
                    'amount'            => $payment->amount->value,
                    'currency_code'     => $invoice->currency_code,
                    'remaining_balance' => 0,
                    'payment_method_id' => ($onlinePaymentMethod != null) ? $onlinePaymentMethod->id : config('fi.onlinePaymentMethod'),
                    'paid_at'           => Carbon::now()->format('Y-m-d'),
                ]);

                if ($fiPayment)
                {
                    $paymentInvoice                         = new PaymentInvoice();
                    $paymentInvoice->payment_id             = $fiPayment->id;
                    $paymentInvoice->invoice_id             = $invoice->id;
                    $paymentInvoice->invoice_amount_paid    = $payment->amount->value;
                    $paymentInvoice->convenience_charges    = $mollieConvenienceCharges;
                    $paymentInvoice->payment_terms_discount = $paymentTermsDiscount;
                    $paymentInvoice->installment_number     = $onlinePaymentChargesDetail['installment'];

                    $paymentInvoice->save();

                    if (!empty($quoteUrlKey) && $quoteUrlKey != 'fi')
                    {
                        $quote = Quote::whereUrlKey($quoteUrlKey)->first();
                        if (isset($quote->id))
                        {
                            event(new QuoteApproved($quote, 'quote_payment_successfully'));
                        }
                    }

                    if ($fiPayment->client->should_email_payment_receipt)
                    {
                        event(new PaymentEmailed($fiPayment));
                    }
                }
                if (isset($fiPayment->id) && isset($payment->id))
                {
                    MerchantPayment::saveByKey($this->getName(), $fiPayment->id, 'id', $payment->id);
                }
            }

        }
        catch (\Exception $e)
        {
            Log::error($e->getMessage());

            if ($quoteUrlKey != '')
            {
                if ($modal == 'pi' && class_exists(\Addons\ProformaInvoice\Models\ProformaInvoice::class) && config('proforma_invoice_enabled') == true)
                {
                    $proformaInvoice = \Addons\ProformaInvoice\Models\ProformaInvoice::where('url_key', $quoteUrlKey)->first();
                    $proformaInvoice->invoice->delete();
                    return redirect()->route('clientCenter.public.proformaInvoice.show', [$quoteUrlKey, $proformaInvoice->token])
                        ->with('alert', trans('fi.payment_problem'));
                }
                else
                {
                    $quote = Quote::where('url_key', $quoteUrlKey)->first();
                    $quote->invoice->delete();
                    return redirect()->route('clientCenter.public.quote.show', [$quoteUrlKey, $quote->token])
                        ->with('alert', trans('fi.payment_problem'));
                }
            }

            return redirect()->route('clientCenter.public.invoice.show', [$invoice->url_key, $invoice->token])
                ->with('error', trans('fi.payment_problem'));
        }
    }
}