<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Invoices\Requests;

use FI\Modules\Invoices\Models\Invoice;
use FI\Modules\Invoices\Rule\InvoiceStatusCheck;
use FI\Requests\RemoveTagsRule;
use FI\Support\NumberFormatter;
use FI\Traits\CustomFieldValidator;

class InvoiceUpdateRequest extends InvoiceStoreRequest
{
    use CustomFieldValidator;

    private $customFieldType         = 'invoices';
    private $lineItemCustomFieldType = 'invoice_items';

    public function prepareForValidation()
    {
        $request  = $this->all();
        $invoice  = Invoice::find($request['invoice_id']);
        $currency = isset($invoice->currency_code) ? $invoice->currency : NULL;

        if (isset($request['items']))
        {
            foreach ($request['items'] as $key => $item)
            {
                $request['items'][$key]['quantity'] = NumberFormatter::unformat($item['quantity'], $currency);
                $request['items'][$key]['price']    = NumberFormatter::unformat($item['price'], $currency);

                if (isset($request['items'][$key]['discount']))
                {
                    $request['items'][$key]['discount'] = NumberFormatter::unformat($item['discount'], $currency);
                }
            }
        }

        $this->replace($request);
    }

    public function rules()
    {
        return [
            'summary'             => ['max:255', new RemoveTagsRule],
            'invoice_date'        => 'required',
            'type'                => 'required',
            'due_at'              => 'required',
            'number'              => ['required', new RemoveTagsRule],
            'exchange_rate'       => 'required|numeric',
            'template'            => 'required',
            'items'               => 'required',
            'payment_terms_id'    => ['required_if:type,invoice'],
            'items.*.name'        => 'required',
            'items.*.quantity'    => 'required_with:items.*.price,items.*.name|numeric',
            'items.*.price'       => ['required_with:items.*.name,items.*.quantity', 'numeric'],
            'items.*.description' => [new RemoveTagsRule],
            'terms'               => [new RemoveTagsRule],
            'footer'              => [new RemoveTagsRule],
            'items.*.discount'    => function ($attribute, $value, $fail)
            {
                $index = explode('.', $attribute);
                if (isset($this['items'][$index[1]]['discount_type']) && $this['items'][$index[1]]['discount_type'] == 'percentage')
                {
                    if ($value > 99.99)
                    {
                        $fail(trans('fi.line_item_discount_limit'));
                    }
                    return true;
                }
                return true;
            },
            'status'              => new InvoiceStatusCheck($this['invoice_id']),
            'discount'            => function ($attribute, $value, $fail)
            {
                if ($this->discount_type == 'percentage')
                {
                    if ($value > 99.99)
                    {
                        $fail(trans('fi.discount_limit'));
                    }
                }
            },
            'tags.*'              => [new RemoveTagsRule]
        ];
    }

    public function messages()
    {
        return [
            'items.*.price' => trans('fi.price_not_valid'),
        ];
    }
}