<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Invoices\Models;

use FI\Modules\Currencies\Support\CurrencyConverterFactory;
use FI\Modules\CustomFields\Models\InvoiceCustom;
use FI\Modules\DocumentNumberSchemes\Models\DocumentNumberScheme;
use FI\Modules\Expenses\Models\Expense;
use FI\Modules\Invoices\Events\AddTransition;
use FI\Modules\Invoices\Support\InvoiceCalculate;
use FI\Modules\Mru\Models\Mru;
use FI\Modules\Payments\Models\Payment;
use FI\Modules\Quotes\Models\Quote;
use FI\Support\DateFormatter;
use Illuminate\Support\Str;

class InvoiceObserver
{
    public function created(Invoice $invoice)
    {
        // Create the empty invoice amount record.
        $invoiceCalculate = new InvoiceCalculate();
        $invoiceCalculate->calculate($invoice);

        // Create the custom invoice record.
        $invoice->custom()->save(new InvoiceCustom());

        // Update the client's status to customer
        $client       = $invoice->client;
        $client->type = 'customer';
        $client->save();

        if ($invoice->type == 'credit_memo')
        {
            event(new AddTransition($invoice, 'credit_memo_created'));
        }
        else
        {
            if ($invoice->recurring_invoice_id)
            {
                event(new AddTransition($invoice, 'created_from_recurring'));
            }
            else
            {
                event(new AddTransition($invoice, 'created'));
            }
        }

        if (($invoice->client->online_payment_processing_fee == 'yes' && config('fi.enableOppFees', 'no') == 1) || ($invoice->client->online_payment_processing_fee == 'default'  && config('fi.enableOppFees', 'no') == 1))
        {
            $invoice->online_payment_processing_fee = 'yes';
            $invoice->save();
        }
    }

    public function creating(Invoice $invoice)
    {
        if (!$invoice->user_id)
        {
            $invoice->user_id = auth()->user()->id;
        }

        if (!$invoice->invoice_date)
        {
            $invoice->invoice_date = date('Y-m-d');
        }

        if (!$invoice->due_at)
        {
            if ($invoice->type == 'credit_memo')
            {
                $invoice->due_at = DateFormatter::incrementDate($invoice->invoice_date, 4, 1);
            }
            else
            {
                $invoice->due_at = Invoice::paymentTermDueDate($invoice->payment_terms_id, $invoice);
            }
        }

        if (!$invoice->company_profile_id)
        {
            $invoice->company_profile_id = config('fi.defaultCompanyProfile');
        }

        if (!$invoice->document_number_scheme_id)
        {
            $invoice->document_number_scheme_id = config('fi.invoiceGroup');
        }

        if (!$invoice->number)
        {
            $invoice->number = DocumentNumberScheme::generateNumber($invoice->document_number_scheme_id, $invoice->client->invoice_prefix);
        }

        if (!isset($invoice->terms))
        {
            $invoice->terms = config('fi.invoiceTerms');
        }

        if (!isset($invoice->footer))
        {
            $invoice->footer = config('fi.invoiceFooter');
        }

        if (!$invoice->status)
        {
            $invoice->status = 'draft';
        }

        if (!$invoice->currency_code)
        {
            $invoice->currency_code = $invoice->client->currency_code;
        }

        if (!$invoice->template)
        {
            $invoice->template = $invoice->companyProfile->invoice_template;
        }

        if ($invoice->currency_code == config('fi.baseCurrency'))
        {
            $invoice->exchange_rate = 1;
        }
        else if (!$invoice->exchange_rate)
        {
            $currencyConverter      = CurrencyConverterFactory::create();
            $invoice->exchange_rate = $currencyConverter->convert(config('fi.baseCurrency'), $invoice->currency_code);
        }

        $invoice->url_key = Str::random(32);
    }

    public function updating(Invoice $invoice)
    {
        if (($invoice->isDirty('payment_terms_id') && $invoice->type == 'invoice') || ($invoice->isDirty('invoice_date') && $invoice->type == 'invoice'))
        {
            if ($invoice->payment_terms_id != null)
            {
                $invoice->due_at = Invoice::paymentTermDueDate($invoice->payment_terms_id, $invoice);
            }
            else
            {
                $invoice->due_at = DateFormatter::incrementDateByDays($invoice->invoice_date, invoicesDueAfter());
            }
        }
    }

    public function deleted(Invoice $invoice)
    {
        foreach ($invoice->items as $item)
        {
            if (config('commission_enabled'))
            {
                foreach ($item->commissions as $commission)
                {
                    $commission->delete();
                }
            }

            $item->delete();
        }

        foreach ($invoice->payments as $payment)
        {
            $payment->delete();
        }

        foreach ($invoice->activities as $activity)
        {
            $activity->delete();
        }

        foreach ($invoice->mailQueue as $mailQueue)
        {
            $mailQueue->delete();
        }

        foreach ($invoice->notes as $note)
        {
            $note->delete();
        }

        $invoice->custom()->delete();

        $invoice->amount()->delete();

        if ($invoice->type == 'credit_memo')
        {
            $payments = Payment::with('paymentInvoice')->whereCreditMemoId($invoice->id)->get();
            foreach ($payments as $payment)
            {
                foreach ($payment->paymentInvoice as $paymentInvoice)
                {
                    $paymentInvoice->delete();
                }
            }
        }

        $quote = Quote::where('invoice_id', $invoice->id)->first();

        if (isset($quote->notifications))
        {
            foreach ($quote->notifications as $notification)
            {
                $notification->delete();
            }
            $quote->status     = 'draft';
            $quote->invoice_id = 0;
            $quote->save();
        }
        if (class_exists(\Addons\ProformaInvoice\Models\ProformaInvoice::class) && config('proforma_invoice_enabled') == true)
        {
            $proformaInvoice = \Addons\ProformaInvoice\Models\ProformaInvoice::where('invoice_id', $invoice->id)->first();

            if (isset($proformaInvoice->notifications))
            {
                foreach ($proformaInvoice->notifications as $notification)
                {
                    $notification->delete();
                }
                $proformaInvoice->status     = 'draft';
                $proformaInvoice->invoice_id = 0;
                $proformaInvoice->save();
            }
        }
        Expense::where('invoice_id', $invoice->id)->update(['invoice_id' => 0]);

        Mru::whereModule('invoices')->whereElementId($invoice->id)->delete();
    }
}