<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Expenses\Requests;

use FI\Requests\RemoveTagsRule;
use FI\Support\NumberFormatter;
use FI\Traits\CustomFieldValidator;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class ExpenseRequest extends FormRequest
{
    use CustomFieldValidator;

    private $customFieldType = 'expenses';

    public function authorize()
    {
        return true;
    }

    public function attributes()
    {
        return [
            'user_id'            => trans('fi.user'),
            'company_profile_id' => trans('fi.company_profile'),
            'vendor_id'          => trans('fi.vendor'),
            'expense_date'       => trans('fi.date'),
            'category_name'      => trans('fi.category'),
            'description'        => trans('fi.description'),
            'amount'             => trans('fi.amount'),
        ];
    }

    public function prepareForValidation()
    {
        $request = $this->all();

        if (isset($request['amount']))
        {
            $request['amount'] = NumberFormatter::unformat($request['amount']);
        }
        else
        {
            $request['amount'] = 0.00;
        }

        if (!isset($request['tax_paid']))
        {
            $request['tax_paid'] = null;
        }

        if (!isset($request['tax']))
        {
            $request['tax'] = null;
        }

        $this->replace($request);
    }

    public function rules()
    {
        return [
            'user_id'            => 'required',
            'company_profile_id' => 'required',
            'vendor_name'        => 'required',
            'expense_date'       => 'required',
            'category_name'      => 'required',
            'description'        => ['max:255', new RemoveTagsRule],
            'type'               => 'required',
            'amount'             => Rule::requiredIf(function ()
                {
                    $type = $this->input('type', '');
                    return ($type == 'remittance_to_tax_authority') ? false : true;
                }) . '|numeric',
            'tax'                => [new RemoveTagsRule],
        ];
    }
}