<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Currencies\Models;

use FI\Modules\Clients\Models\Client;
use FI\Modules\Invoices\Models\Invoice;
use FI\Modules\Payments\Models\Payment;
use FI\Modules\Quotes\Models\Quote;
use FI\Modules\RecurringInvoices\Models\RecurringInvoice;
use FI\Traits\Sortable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Schema;

class Currency extends Model
{
    use Sortable;

    protected $table = 'currencies';

    protected $sortable = ['code', 'name', 'symbol', 'placement', 'decimal', 'thousands'];

    /**
     * Guarded properties
     * @var array
     */
    protected $guarded = ['id'];

    public static function getList()
    {
        return Cache::rememberForever('getListCurrenciesCache', function ()
        {
            return self::orderBy('name')->pluck('name', 'code')->all();
        });
    }

    public static function getListBySymbol()
    {
        return self::orderBy('name')->pluck('symbol')->toArray();
    }

    public static function getByCode($code)
    {
        return self::where('code', $code)->first();
    }

    public function getInUseAttribute()
    {
        $listForUsedCurrency = [];

        if ($this->code == config('fi.baseCurrency'))
        {
            return trans('fi.base_currency_not_deleted');
        }

        if (Client::where('currency_code', '=', $this->code)->count())
        {
            $listForUsedCurrency[] = trans('fi.clients');
        }

        if (Quote::where('currency_code', '=', $this->code)->count())
        {
            $listForUsedCurrency[] = trans('fi.quotes');
        }

        if (Invoice::where('currency_code', '=', $this->code)->count())
        {
            $listForUsedCurrency[] = trans('fi.invoices');
        }

        if (RecurringInvoice::where('currency_code', '=', $this->code)->count())
        {
            $listForUsedCurrency[] = trans('fi.subscriptions');
        }

        if (Payment::where('currency_code', '=', $this->code)->count())
        {
            $listForUsedCurrency[] = trans('fi.payments');
        }

        if (Schema::hasTable('proforma_invoices'))
        {
            if (\Addons\ProformaInvoice\Models\ProformaInvoice::where('currency_code', '=', $this->code)->count())
            {
                $listForUsedCurrency[] = trans('ProformaInvoice::lang.proforma_invoices');
            }
        }

        if (count($listForUsedCurrency) > 0)
        {
            return trans('fi.used_currency_modal_error', ['modal_names' => implode(', ', $listForUsedCurrency)]);
        }

        // Must also check to see if payments table has the currency in use.
       
        // Must also check to see if proforma invoices have the currency in use (if ProForma Invoices Add-on is installed)

        return false;
    }

    public function getSystemBaseCurrencyAttribute()
    {
        if ($this->code == config('fi.baseCurrency'))
        {
            return true;
        }

        return false;
    }

    public function getFormattedPlacementAttribute()
    {
        return ($this->placement == 'before') ? trans('fi.before_amount') : trans('fi.after_amount');
    }

    public static function getSeparators()
    {
        return [
            '&#32;'   => trans('fi.space'),
            '&#39;'   => trans('fi.apostrophe'),
            '&#44;'   => trans('fi.comma'),
            '&#46;'   => trans('fi.fullstop'),
            '&#183;'  => trans('fi.middledot'),
            '&#8201;' => trans('fi.thinspace'),
            '&#8239;' => trans('fi.narrow-no-break-space'),
            '&#729;'  => trans('fi.dotabove'),
            '&#1643;' => trans('fi.arabic-decimal'),
            '&#1644;' => trans('fi.arabic-thousand'),
            '&#9110;' => trans('fi.decimal-separator-key-symbol'),
        ];
    }
}
