<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Contacts\Controllers;

use Carbon\Carbon;
use FI\Http\Controllers\Controller;
use FI\Modules\Contacts\Models\Contact;
use FI\Modules\Contacts\Models\ContactTag;
use FI\Modules\Contacts\Requests\ContactRequest;
use FI\Modules\Tags\Models\Tag;

class ContactController extends Controller
{
    public function index($clientId)
    {
        $contacts = Contact::where('client_id', $clientId)->orderBy('name')->get();
        return view('contacts._contacts')
            ->with('clientId', $clientId)
            ->with('contacts', $contacts)->render();
    }

    public function create($clientId)
    {
        return view('contacts._modal_contact')
            ->with('editMode', false)
            ->with('clientId', $clientId)
            ->with('defaultCountry', Contact::getContactDefaultCountry($clientId))
            ->with('contactTitle', ['' => trans('fi.select_title')] + Contact::getContactTitle())
            ->with('submitRoute', route('clients.contacts.store', [$clientId]))
            ->with('contactTags', Tag::whereTagEntity('contact')->pluck('name', 'name'))
            ->with('contactSelectedTags', [])
            ->with('getDefaultStatusListForToBccAndCc', Contact::getDefaultStatusListForToBccAndCc());

    }

    public function store(ContactRequest $request, $clientId)
    {
        $input = $request->except('tags');

        $contact = Contact::create($input);

        $manageTags = manageTags($contact, 'contact_tag_updated', 'contact_tag_deleted', 'Contacts');

        $tags    = isset($manageTags) ? $manageTags : $request->input('tags', []);
        $tag_ids = [];

        foreach ($tags as $tag)
        {
            $tag = Tag::firstOrNew(['name' => $tag, 'tag_entity' => 'contact'])->fill(['name' => $tag, 'tag_entity' => 'contact']);

            $tag->save();

            $tag_ids[] = $tag->id;
        }

        foreach ($tag_ids as $tag_id)
        {
            $contact->tags()->create(['contact_id' => $contact->id, 'tag_id' => $tag_id]);
        }

        return $this->index($clientId);
    }

    public function edit($clientId, $id)
    {

        $contact      = Contact::find($id);
        $selectedTags = [];

        foreach ($contact->tags as $tagDetail)
        {
            $selectedTags[] = $tagDetail->tag->name;
        }

        return view('contacts._modal_contact')
            ->with('editMode', true)
            ->with('clientId', $clientId)
            ->with('contactTitle', ['' => trans('fi.select_title')] + Contact::getContactTitle())
            ->with('submitRoute', route('clients.contacts.update', [$clientId, $id]))
            ->with('defaultCountry', Contact::getContactDefaultCountry($clientId))
            ->with('contact', $contact)
            ->with('contactTags', Tag::whereTagEntity('contact')->pluck('name', 'name'))
            ->with('contactSelectedTags', $selectedTags)
            ->with('getDefaultStatusListForToBccAndCc', Contact::getDefaultStatusListForToBccAndCc());
    }

    public function update(ContactRequest $request, $clientId, $id)
    {
        $contact = Contact::find($id);
        $input   = $request->except('tags');
        $contact->fill($input);
        $contact->save();

        $manageTags = manageTags($contact, 'contact_tag_updated', 'contact_tag_deleted', 'Contacts');

        $tags    = isset($manageTags) ? $manageTags : $request->input('tags', []);
        $tag_ids = [];

        foreach ($tags as $tag)
        {
            $tag = Tag::firstOrNew(['name' => $tag, 'tag_entity' => 'contact'])->fill(['name' => $tag, 'tag_entity' => 'contact']);

            $tag->save();

            $tag_ids[] = $tag->id;
        }

        foreach ($tag_ids as $tag_id)
        {
            ContactTag::firstOrNew(['contact_id' => $contact->id, 'tag_id' => $tag_id])->fill(['contact_id' => $contact->id, 'tag_id' => $tag_id, 'created_at' => Carbon::now(), 'updated_at' => Carbon::now()])->save();
        }

        return $this->index($clientId);
    }

    public function delete($clientId)
    {
        Contact::destroy(request('id'));

        return response()->json([
            'message' => trans('fi.record_successfully_deleted'),
            'html'    => $this->index($clientId)
        ], 200);
    }

    public function updateDefault($clientId)
    {
        $contact = Contact::find(request('id'));

        switch (request('default'))
        {
            case 'to':
                $data = [
                    'default_to'  => ($contact->default_to) ? 0 : 1,
                    'default_cc'  => 0,
                    'default_bcc' => 0,
                ];
                break;
            case 'cc':
                $data = [
                    'default_to'  => 0,
                    'default_cc'  => ($contact->default_cc) ? 0 : 1,
                    'default_bcc' => 0,
                ];
                break;
            case 'bcc':
                $data = [
                    'default_to'  => 0,
                    'default_cc'  => 0,
                    'default_bcc' => ($contact->default_bcc) ? 0 : 1,
                ];
                break;
        }

        $contact->fill($data);
        $contact->save();

        return $this->index($clientId);
    }

    public function deleteContactModal()
    {
        try
        {
            return view('contacts._modal_delete_contact')->with('contactId', request('id'))->with('url', request('action'))->with('returnURL', request('returnURL'))->with('modalName', request('modalName'))->with('isReload', request('isReload'));
        }
        catch (\Exception $e)
        {
            return response()->json(['success' => false, 'message' => ($e->getCode() == 0) ? trans('fi.modal_not_found') : $e->getMessage()], 400);
        }
    }
}