<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Clients\Controllers;

use FI\Http\Controllers\Controller;
use FI\Modules\Attachments\Models\Attachment;
use FI\Modules\Clients\Events\AddTransitionMerge;
use FI\Modules\Clients\Models\Client;
use FI\Modules\Clients\Models\ClientTag;
use FI\Modules\Clients\Models\Contact;
use FI\Modules\Clients\Requests\ClientGetAjaxRequest;
use FI\Modules\Clients\Requests\ClientMergeRecordsRequest;
use FI\Modules\CustomFields\Support\CustomFieldsParser;
use FI\Modules\Expenses\Models\Expense;
use FI\Modules\Invoices\Models\Invoice;
use FI\Modules\Merchant\Models\MerchantClient;
use FI\Modules\Mru\Models\Mru;
use FI\Modules\Notes\Models\Note;
use FI\Modules\Notifications\Models\Notification;
use FI\Modules\Payments\Models\Payment;
use FI\Modules\Quotes\Models\Quote;
use FI\Modules\RecurringInvoices\Models\RecurringInvoice;
use FI\Modules\TaskList\Models\Task;
use FI\Modules\Transitions\Models\Transitions;
use FI\Modules\Users\Models\User;
use FI\Traits\ReturnUrl;
use Illuminate\Support\Facades\DB;

class ClientMergeController extends Controller
{
    use ReturnUrl;

    public function mergeRecords()
    {
        try
        {
            return view('clients.merge')->with('returnUrl', $this->getReturnUrl())->with('clients', Client::getDropDownList())->with('id', request('merge_id', null));
        }
        catch (\Exception $e)
        {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 400);
        }
    }

    public function ajaxGetClientRecord(ClientGetAjaxRequest $request)
    {
        try
        {
            $client = Client::with(['tags.tag', 'contacts'])->find($request->id);
            return view('clients._ajax_merge_client_record')
                ->with('clients', Client::getDropDownList())
                ->with('basicArray', ['phone' => 'PhoneNumber', 'mobile' => 'MobileNumber', 'fax' => 'FaxNumber', 'web' => 'WebAddress', 'social_media_url' => 'SocialMediaUrl'])
                ->with('settingArray', ['invoice_prefix' => 'InvoicePrefix', 'automatic_email_payment_receipt' => 'AutomaticEmailPaymentReceipt', 'automatic_email_on_recur' => 'AutomaticEmailOnRecurringInvoice', 'currency_code' => 'DefaultCurrency', 'language' => 'Language', 'timezone' => 'Timezone', 'online_payment_processing_fee' => 'OnlinePaymentProcessingFee', 'allow_child_accounts' => 'AllowChildAccounts', 'third_party_bill_payer' => 'ThirdPartyBillPayer',])
                ->with('typeLabels', ['lead' => 'badge-warning', 'prospect' => 'badge-danger', 'customer' => 'badge-success', 'affiliate' => 'badge-info', 'other' => 'badge-secondary'])
                ->with('customFields', CustomFieldsParser::getFields('clients'))
                ->with('surrogate', request('surrogate', false))
                ->with('dataSurrogateOne', ['ignore' => null, 'overwrite' => trans('fi.overwrite')])
                ->with('dataSurrogateTwo', ['ignore' => null, 'overwrite' => trans('fi.overwrite'), 'add_to' => trans('fi.add_to')])
                ->with('dataSurrogateThree', ['ignore' => null, 'add_to' => trans('fi.add_to')])
                ->with('client', $client)->render();
        }
        catch (\Exception $e)
        {
            return response()->json(['success' => false, 'message' => ($e->getCode() == 0) ? trans('fi.modal_not_found') : $e->getMessage()], 400);
        }

    }

    public function mergeArrayPrepare($array)
    {
        $uniqueValues = array_unique($array);
        $newArray     = [];
        foreach ($uniqueValues as $value)
        {
            $filteredArray = array_filter($array, function ($element) use ($value)
            {
                return $element === $value;
            }, ARRAY_FILTER_USE_BOTH);

            $newArray[$value] = $filteredArray;
        }
        return $newArray;
    }

    public function mergeCustomFields($clientTo, $separateCustomArrays)
    {
        if (isset($separateCustomArrays['add_to']) && count($separateCustomArrays['add_to']) > 0)
        {
            $this->mergeClientCustomFieldsAddTo($clientTo, $separateCustomArrays['add_to']);
        }
        if (isset($separateCustomArrays['overwrite']) && count($separateCustomArrays['overwrite']) > 0)
        {
            $this->mergeClientCustomFieldsOverwrite($clientTo, $separateCustomArrays['overwrite']);
        }
        $clientTo->custom->save();
    }

    public function mergeClientCustomFieldsOverwrite($clientTo, $customOverwrite)
    {
        foreach ($customOverwrite as $custom => $value)
        {
            $keyAndValue                       = explode('##', $custom);
            $clientTo->custom[$keyAndValue[0]] = $keyAndValue[1];
        }
    }

    public function mergeClientCustomFieldsAddTo($clientTo, $customAddTo)
    {
        foreach ($customAddTo as $custom => $value)
        {
            $keyAndValue = explode('##', $custom);
            $field       = CustomFieldsParser::getFieldByColumnName('clients', $keyAndValue[0]);

            if ($field->field_label == 'selestteg')
            {
                if ($clientTo->custom[$keyAndValue[0]] != null)
                {
                    $tags                              = $clientTo->custom[$keyAndValue[0]] . ',' . $keyAndValue[1];
                    $mergeTags                         = implode(',', array_unique(explode(',', $tags)));
                    $clientTo->custom[$keyAndValue[0]] = $mergeTags;
                }
                else
                {
                    $clientTo->custom[$keyAndValue[0]] = $keyAndValue[1];
                }
            }
            else
            {
                if ($clientTo->custom[$keyAndValue[0]] != null)
                {
                    $clientTo->custom[$keyAndValue[0]] = $clientTo->custom[$keyAndValue[0]] . "\n" . $keyAndValue[1];
                }
                else
                {
                    $clientTo->custom[$keyAndValue[0]] = $keyAndValue[1];
                }
            }
        }
    }

    public function mergeAllTransactions($clientTo, $clientFrom, $separateCustomArrays)
    {
        try
        {
            $index                = 0;
            $mergeHistory         = [];
            $quoteIds             = $clientFrom->quotes->pluck('id');
            $invoiceIds           = $clientFrom->invoices->pluck('id');
            $recurringInvoicesIds = $clientFrom->recurringInvoices->pluck('id');
            $paymentsIds          = $clientFrom->payments->pluck('id');
            $expensesIds          = $clientFrom->expenses->pluck('id');
            $tasksIds             = $clientFrom->tasks->pluck('id');
            $notesIds             = $clientFrom->notes->pluck('id');
            $attachmentsIds       = $clientFrom->attachments->pluck('id');
            $contactsIds          = $clientFrom->contacts->pluck('id');
            $transitionsOneIds    = Transitions::where('transitionable_id', $clientFrom->id)->orWhere('client_id', $clientFrom->id)->where('transitionable_type', 'FI\Modules\Clients\Models\Client')->pluck('id');
            $transitionsTwoIds    = Transitions::where('client_id', $clientFrom->id)->where('transitionable_type', '<>', 'FI\Modules\Clients\Models\Client')->pluck('id');
            $notificationIds      = Notification::where('notifiable_type', 'FI\Modules\Clients\Models\Client')->where('notifiable_id', $clientFrom->id)->pluck('id');
            if ($clientTo->merge_history != null)
            {
                $mergeHistory = json_decode($clientTo->merge_history, true);
                $index        = count($mergeHistory);
            }

            if (count($separateCustomArrays) != 0)
            {
                $mergeHistory[$index]['custom'] = ['clientTo' => $clientTo->custom, 'clientFrom' => $clientFrom->custom];
            }
            if ($attachmentsIds->count() > 0)
            {
                Attachment::whereIn('id', $attachmentsIds)->update(['attachable_id' => $clientTo->id]);
                $mergeHistory[$index]['attachments'] = $attachmentsIds;
            }
            if ($quoteIds->count() > 0)
            {
                Quote::whereIn('id', $quoteIds)->update(['client_id' => $clientTo->id]);
                $mergeHistory[$index]['quotes'] = $quoteIds;
            }
            if ($invoiceIds->count() > 0)
            {
                Invoice::whereIn('id', $invoiceIds)->update(['client_id' => $clientTo->id]);
                $mergeHistory[$index]['invoices'] = $invoiceIds;
            }

            if ($recurringInvoicesIds->count() > 0)
            {
                RecurringInvoice::whereIn('id', $recurringInvoicesIds)->update(['client_id' => $clientTo->id]);
                $mergeHistory[$index]['recurring_invoices'] = $recurringInvoicesIds;
            }

            if ($paymentsIds->count() > 0)
            {
                Payment::whereIn('id', $paymentsIds)->update(['client_id' => $clientTo->id]);
                $mergeHistory[$index]['payments'] = $paymentsIds;
            }

            if ($expensesIds->count() > 0)
            {
                Expense::whereIn('id', $expensesIds)->update(['client_id' => $clientTo->id]);
                $mergeHistory[$index]['expenses'] = $expensesIds;
            }

            if ($tasksIds->count() > 0)
            {
                Task::whereIn('id', $tasksIds)->update(['client_id' => $clientTo->id]);
                $mergeHistory[$index]['tasks'] = $tasksIds;
            }

            if ($notesIds->count() > 0)
            {
                Note::whereIn('id', $notesIds)->update(['notable_id' => $clientTo->id]);
                $mergeHistory[$index]['notes'] = $notesIds;
            }

            if ($contactsIds->count() > 0)
            {
                Contact::whereIn('id', $contactsIds)->update(['client_id' => $clientTo->id]);
                $mergeHistory[$index]['contacts'] = $contactsIds;
            }

            if ($transitionsOneIds->count() > 0 || $transitionsTwoIds->count() > 0)
            {
                Transitions::whereIn('id', $transitionsOneIds)->update(['client_id' => $clientTo->id, 'transitionable_id' => $clientTo->id]);
                Transitions::whereIn('id', $transitionsTwoIds)->update(['client_id' => $clientTo->id]);
                $mergeHistory[$index]['transitions'] = array_merge($transitionsTwoIds->toArray(), $transitionsOneIds->toArray());
            }
            if ($notificationIds->count() > 0)
            {
                Notification::where('notifiable_type', 'FI\Modules\Clients\Models\Client')->where('notifiable_id', $clientFrom->id)->update(['notifiable_id' => $clientTo->id]);
                $mergeHistory[$index]['notification'] = $notificationIds;
            }

            $arrayClientTo   = Client::find($clientTo->id)->toArray();
            $arrayClientFrom = Client::find($clientFrom->id)->toArray();
            unset($arrayClientTo['merge_history']);
            unset($arrayClientFrom['merge_history']);
            $mergeHistory[$index]['client'] = ['clientTo' => $arrayClientTo, 'clientFrom' => $arrayClientFrom];
            $clientTo->merge_history        = json_encode($mergeHistory, true);
        }
        catch (\Exception $e)
        {
            throw new \Exception($e->getMessage());
        }
    }

    public function mergeClientOverwriteAction($clientTo, $clientFrom, $overWrite)
    {
        $addressNeeds = ['address', 'city', 'state', 'zip', 'country'];

        foreach ($overWrite as $key => $value)
        {
            if ($key == 'tags')
            {
                if (isset($clientFrom->tags) && $clientFrom->tags->count() > 0)
                {
                    $clientTo->deleteTags($clientTo);
                    ClientTag::where('client_id', $clientFrom->id)->update(['client_id' => $clientTo->id]);
                }
            }
            elseif ($key == 'address')
            {
                foreach ($addressNeeds as $addressValue)
                {
                    $clientTo[$addressValue] = $clientFrom[$addressValue];
                }
            }
            else
            {
                $clientTo[$key] = $clientFrom[$key];
            }
        }
    }

    public function mergeClientAddToAction($clientTo, $clientFrom, $addTo)
    {
        foreach ($addTo as $key => $value)
        {
            if ($key == 'tags')
            {
                if (isset($clientFrom->tags) && $clientFrom->tags->count() > 0)
                {
                    $clientFromTagsIds = ClientTag::where('client_id', $clientFrom->id)->pluck('tag_id')->toArray();
                    $clientToTagsIds   = ClientTag::where('client_id', $clientTo->id)->pluck('tag_id')->toArray();
                    $arrayDiff         = array_diff($clientFromTagsIds, $clientToTagsIds);
                    ClientTag::where('client_id', $clientFrom->id)->whereIn('tag_id', $arrayDiff)->update(['client_id' => $clientTo->id]);
                }
            }
            else
            {
                $clientTo[$key] = $clientTo[$key] . "\n" . $clientFrom[$key];
            }
        }
    }

    public function mergeModal()
    {
        try
        {
            $parentClient         = Client::getParentClientsForMerge(request('client_from', 0));
            $childClients         = Client::getChildClients(request('client_from', 0));
            $thirdPartyBillPayers = Client::getThirdPartyBillPayers(request('client_from', 0));
            $invoicesPaidBy       = Client::getInvoicesPaidByClientsForMerge(request('client_from', 0));
            $dynamicWidth         = (count($parentClient) > 0 && count($invoicesPaidBy) > 0) ? 6 : 12;

            if ($childClients->count() > 0 || count($parentClient) > 0 || count($invoicesPaidBy) > 0 || count($thirdPartyBillPayers) > 0)
            {
                return view('clients._modal_merge_child_account_worn')
                    ->with('url', null)
                    ->with('parentClient', $parentClient)
                    ->with('thirdPartyBillPayers', $thirdPartyBillPayers)
                    ->with('invoicesPaidBy', $invoicesPaidBy)
                    ->with('dynamicWidth', $dynamicWidth)
                    ->with('data', request()->all())
                    ->with('childClients', $childClients);
            }

            return view('clients._modal_merge')->with('url', null)->with('data', request()->all());
        }
        catch (\Exception $e)
        {
            return response()->json(['success' => false, 'message' => ($e->getCode() == 0) ? trans('fi.modal_not_found') : $e->getMessage()], 400);
        }
    }

    public function mergeClientRecordsSet(ClientMergeRecordsRequest $request)
    {
        try
        {
            DB::beginTransaction();

            $clientTo   = Client::with(['tags.tag', 'contacts'])->find($request->client_to);
            $clientFrom = Client::with(['tags.tag', 'contacts'])->find($request->client_from);

            $staticTotal  = Client::countStaticRelationship();
            $dynamicTotal = $clientFrom->countDynamicRelationship();

            if ($staticTotal != $dynamicTotal)
            {
                throw new \Exception(trans('fi.merge_relationships_warning'));
            }

            $separateCustomArrays = [];

            $separateArrays = $this->mergeArrayPrepare($request->action);

            if ($request->custom != null)
            {
                $separateCustomArrays = $this->mergeArrayPrepare($request->custom);
            }

            $this->mergeAllTransactions($clientTo, $clientFrom, $separateCustomArrays);

            if (isset($separateArrays['add_to']) && count($separateArrays['add_to']) > 0)
            {
                $this->mergeClientAddToAction($clientTo, $clientFrom, $separateArrays['add_to']);
            }
            if (isset($separateArrays['overwrite']) && count($separateArrays['overwrite']) > 0)
            {
                $this->mergeClientOverwriteAction($clientTo, $clientFrom, $separateArrays['overwrite']);
            }

            $client_from = Client::whereId($request->client_from)->first();
            Client::reassignChildAccounts($client_from->id, $clientTo->id);
            Client::reassignThirdPartyBillPayer($client_from->id, $clientTo->id);
            User::where('user_type', 'client')->where('client_id', $client_from->id)->delete();
            Mru::where('module','clients')->where('element_id',$client_from->id)->update(['element_id' => $clientTo->id]);
            MerchantClient::where('client_id',$client_from->id)->update(['client_id' => $clientTo->id]);
            event(new AddTransitionMerge($clientTo, $client_from, 'merge'));

            $client_from->delete();

            $clientTo->save();
            $this->mergeCustomFields($clientTo, $separateCustomArrays);

            DB::commit();

            return response()->json(['success' => true, 'redirectUrl' => route('clients.show', ['id' => $clientTo->id]), 'message' => trans('fi.record_successfully_merged')], 200);
        }
        catch (\Exception $e)
        {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => $e->getMessage()], 400);
        }
    }
}
