<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Setup\Controllers;

use Exception;
use FI\Http\Controllers\Controller;
use FI\Modules\Addons\Models\Addon;
use FI\Modules\Attachments\Models\Attachment;
use FI\Modules\CompanyProfiles\Models\CompanyProfile;
use FI\Modules\Countries\Models\Country;
use FI\Modules\Settings\Models\Setting;
use FI\Modules\Setup\Events\AddNotification;
use FI\Modules\Setup\Requests\KeyVerificationRequest;
use FI\Modules\Setup\Requests\LicenseRequest;
use FI\Modules\Setup\Requests\ProfileRequest;
use FI\Modules\Users\Models\User;
use FI\Support\Migrations;
use FI\Support\UpdateChecker;
use FI\Traits\ReturnUrl;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Throwable;

class SetupController extends Controller
{
    use ReturnUrl;

    private $migrations;

    public function __construct(Migrations $migrations)
    {
        $this->migrations = $migrations;
    }

    public function index()
    {
        if (Schema::hasTable('migrations') && empty($this->migrations->getPendingMigrations(base_path('database/migrations'))))
        {
            return redirect()->route('session.login')->with('alertSuccess', trans('fi.version_upgraded'));
        }

        Cookie::queue(Cookie::forever('old_git_version', config('fi.version')));

        if (Schema::hasTable('addons'))
        {
            $addons = Addon::where('enabled', 1)->pluck('name')->toArray();
        }
        return view('setup.index')
            ->with('addons', isset($addons) ? $addons : [])
            ->with('license', file_get_contents(base_path('LICENSE')));
    }

    public function postIndex(LicenseRequest $request)
    {
        if ($request->addon_enabled)
        {
            $addons = Addon::where('enabled', 1)->pluck('name')->toArray();
            Addon::where('enabled', 1)->update(['enabled' => 0]);
            event(new AddNotification(
                    [
                        'message' => trans('fi.addons_disabled_during_setup', ['addons' => implode(', ', $addons)]),
                        'type'    => 'info',
                    ], 'addon_disabled')
            );
        }
        return redirect()->route('setup.preRequirement');
    }

    public function preRequirement()
    {
        $requirements = self::setupPreRequirementCheckData();

        $error = false;

        foreach ($requirements as $required)
        {
            if ($required['result'] == 0)
            {
                $error = true;
            }
        }

        Session::put('pre_requirement', $error);

        return view('setup.pre_requirement')->with('freeSpace', getFreeSpace())->with('requirements', $requirements)->with('requirementsError', $error);
    }

    public function migration()
    {
        $gitBranch = getCurrentGitBranch();

        $session = Session::get('pre_requirement');

        if (!$session)
        {
            Session::forget('pre_requirement');
            return view('setup.migration')->with('branch', $gitBranch);
        }
        else
        {
            return redirect()->route('setup.preRequirement');
        }
    }

    public function postMigration()
    {
        if ($this->migrations->runMigrations(database_path('migrations')))
        {
            Setting::where('setting_key', 'key')->delete();
            return response()->json([], 200);
        }

        return response()->json(['exception' => $this->migrations->getException()->getMessage() . ' ' . $this->migrations->getException()->getFile() . ' ' . $this->migrations->getException()->getLine()], 400);
    }

    public function attachment()
    {

        $attachmentCount = Attachment::selectRaw('COUNT(CASE WHEN content IS NULL THEN 1 END) as moveContent, COUNT(CASE WHEN content IS NOT NULL THEN 1 END) as totalContent')->first();

        if (!User::where('user_type', '<>', 'system')->count())
        {
            return redirect()->route('setup.account');
        }

        if ($attachmentCount->totalContent == 0)
        {
            return redirect()->route('setup.complete');
        }

        $date    = Carbon::now();
        $license = $date->format('Y-m-d') . '_check_up_success_@_' . $date->format('s:i:h');
        Session::put('attachmentCheckKey', $license);

        return view('setup.attachment')->with('attachmentCount', $attachmentCount)->with('attachmentCheckKey', Crypt::encrypt($license));
    }

    public function postAttachment()
    {
        if (!User::where('user_type', '<>', 'system')->count())
        {
            return view('setup.account')->with('countries', Country::getAll());
        }

        try
        {

            if (Crypt::decrypt(request('key')) == Session::get('attachmentCheckKey'))
            {
                Session::put('postAttachmentCheckKey', Crypt::decrypt(request('key')));
                Log::channel('attachment_log')->warning('START move file in media folder from database.');

                $attachments = Attachment::whereNotNull('content')->get();

                Log::channel('attachment_log')->info('Total records is ' . count($attachments));

                foreach ($attachments as $key => $attachment)
                {
                    Log::channel('attachment_log')->warning($key . ' number attachment move media folder from the database.');

                    Log::channel('attachment_log')->info('start : ' . $key . ' record and attachment ID is ' . $attachment->id . ' MODEL is ' . $attachment->attachable_type);

                    try
                    {
                        $modal     = new $attachment->attachable_type;
                        $modalName = $modal->getTable();
                        if ($modalName != '')
                        {
                            $attachmentContent = $attachment->content;
                            $fileExtension     = explode('.', $attachment->filename);
                            $filename          = Str::uuid()->toString() . '.' . end($fileExtension);
                            $path              = $modalName . '/' . $filename;
                            $media_path        = Storage::disk('media')->put($path, $attachmentContent);
                            Log::channel('attachment_log')->info($key . ' ID[' . $attachment->id . '] move successfully attachment and attachment path is => ' . $path);

                            if ($media_path != '' && $media_path != false)
                            {
                                $attachment->media_path = $path;
                                $attachment->content    = null;
                                $attachment->save();
                                Log::channel('attachment_log')->info($key . ' ID[' . $attachment->id . '] database attachment content null set successfully.');
                            }
                        }

                        Log::channel('attachment_log')->info('stop : ' . $key . ' number attachment moved media folder from the database successfully. attachment ID is' . $attachment->id);
                        Log::channel('attachment_log')->warning($key . ' moved successfully.');

                    }
                    catch (Exception|Throwable $e)
                    {
                        Log::channel('attachment_log')->error($key . 'failed number attachment moved media folder from the database with error => ' . $e->getMessage());
                        return response()->json(['exception' => $e->getMessage()], 400);
                    }

                    Log::channel('attachment_log')->warning('STOP : move filed in media folder from database.');
                }

            }
            else
            {
                return response()->json(['exception' => trans('fi.attachment_key_invalid')], 400);
            }

        }
        catch (Exception $e)
        {
            return response()->json(['exception' => $e->getMessage()], 400);
        }

    }

    public function account()
    {
        try
        {
            $attachmentCount = Attachment::selectRaw('COUNT(CASE WHEN content IS NULL THEN 1 END) as moveContent, COUNT(CASE WHEN content IS NOT NULL THEN 1 END) as totalContent')->get()->first();

            if (!User::where('user_type', '<>', 'system')->count())
            {
                return view('setup.account')->with('countries', Country::getAll());
            }

            if (Session::get('attachmentCheckKey') == Session::get('postAttachmentCheckKey') or $attachmentCount->totalContent == 0)
            {
                $companyProfile = CompanyProfile::whereIsDefault(1)->first();
                if (!isset($companyProfile->id))
                {
                    return redirect()->route('setup.companyProfile');
                }
                else
                {
                    return redirect()->route('setup.complete');
                }
            }
            else
            {
                $this->attachment();
            }
        }
        catch (Exception $e)
        {
            return redirect()->back()->with('alert', $e->getMessage());
        }
    }

    public function postAccount(ProfileRequest $request)
    {
        if (!User::where('user_type', '<>', 'system')->count())
        {
            $input = $request->all();

            unset($input['user']['password_confirmation']);

            $user = new User($input['user']);

            $user->password = $input['user']['password'];

            $user->user_type = 'admin';

            $user->save();

            $input['company_profile']['is_default'] = 1;

            $companyProfile = CompanyProfile::create($input['company_profile']);

            Setting::saveByKey('defaultCompanyProfile', $companyProfile->id);

            Setting::saveByKey('mailFromAddress', $user->email);

            User::whereUserType('system')->update(['email' => '', 'password' => '']);

        }

        return redirect()->route('setup.verify.key');
    }

    public function companyProfile()
    {
        return view('setup.company_profile')
            ->with('companyProfiles', CompanyProfile::getList())
            ->with('companyProfileCount', CompanyProfile::count())
            ->with('countries', Country::getAll());
    }

    public function postCompanyProfile(ProfileRequest $request)
    {
        $input = $request->all();

        if ($input['is_create'] == 'edit')
        {
            CompanyProfile::whereId($input['company_profile']['company'])->update(['is_default' => 1]);
        }
        else
        {
            $input['company_profile']['is_default'] = 1;

            $companyProfile = CompanyProfile::create($input['company_profile']);

            Setting::saveByKey('defaultCompanyProfile', $companyProfile->id);

            User::whereUserType('system')->update(['email' => '', 'password' => '']);

        }

        return redirect()->route('setup.complete');

    }

    public function verifyKey()
    {
        $attachmentCount = Attachment::selectRaw('COUNT(CASE WHEN content IS NULL THEN 1 END) as moveContent, COUNT(CASE WHEN content IS NOT NULL THEN 1 END) as totalContent')->get()->first();

        if ($attachmentCount->totalContent != 0)
        {
            return redirect()->route('setup.attachment');
        }

        $companyProfile = CompanyProfile::whereIsDefault(1)->first();

        if ($companyProfile)
        {
            if (!Setting::where('setting_key', 'key')->count())
            {
                return view('setup.verify-key');
            }
            else if (!User::where('user_type', '<>', 'system')->count())
            {
                return redirect()->route('setup.account');
            }
        }
        else
        {
            return redirect()->route('setup.account');
        }
        return redirect()->route('setup.complete');
    }

    public function postVerifyKey(KeyVerificationRequest $request)
    {
        $key        = trim($request->get('key'));
        $settingKey = Setting::where('setting_key', 'key')->first();

        if (isset($settingKey) && strlen($settingKey->setting_value) >= 32)
        {
            return redirect()->route('setup.complete')->with('alertSuccess', trans('fi.key_verified'));
        }
        else
        {
            $companyProfile = CompanyProfile::whereIsDefault(1)->first();
            $mysqlVersion   = DB::select("select version()")[0]->{'version()'} ?? 'UNKNOWN';
            $data           = [
                'key'               => base64_encode($key),
                'domain'            => base64_encode($_SERVER["HTTP_HOST"]),
                'company'           => base64_encode($companyProfile->company),
                'address'           => base64_encode($companyProfile->address),
                'city'              => base64_encode($companyProfile->city),
                'state'             => base64_encode($companyProfile->state),
                'postal_code'       => base64_encode($companyProfile->zip),
                'country'           => base64_encode($companyProfile->country),
                'mail_from_address' => base64_encode(config('fi.mailFromAddress')),
                'mail_from_name'    => base64_encode(config('fi.mailFromName')),
                'php_version'       => base64_encode(PHP_VERSION),
                'db_name'           => base64_encode(config('database.connections.mysql.database')),
                'db_type'           => base64_encode(config('database.default')),
                'db_version'        => base64_encode($mysqlVersion),
                'web_server'        => base64_encode($_SERVER["SERVER_SOFTWARE"]),
                'request_type'      => base64_encode('auto'),
                'version'           => base64_encode(config('fi.version')),
            ];

            $updateChecker = new UpdateChecker;

            $verificationStatus = $updateChecker->keyVerification($data);

            if ($verificationStatus == true)
            {
                Setting::create(
                    [
                        'setting_key'   => 'key',
                        'setting_value' => $key,
                    ]
                );

                Config::write('app.key', $key);

                return redirect()->route('setup.complete')->with('alertSuccess', trans('fi.key_verified'));
            }
        }

        return redirect()->route('setup.verify.key')->with('alert', $updateChecker->message != '' ? $updateChecker->message : trans('fi.invalid_key'));

    }

    public function complete()
    {

        $attachmentCount = Attachment::selectRaw('COUNT(CASE WHEN content IS NULL THEN 1 END) as moveContent, COUNT(CASE WHEN content IS NOT NULL THEN 1 END) as totalContent')->get()->first();

        if ($attachmentCount->totalContent != 0)
        {
            return redirect()->route('setup.attachment');
        }
        if (!Setting::where('setting_key', 'key')->count())
        {
            return redirect()->route('setup.verify.key');
        }
        else if (!User::where('user_type', '<>', 'system')->count())
        {
            return redirect()->route('setup.account');
        }

        return view('setup.complete')->with('oldVersion', Cookie::get('old_git_version', config('fi.version')));
    }

    public function setupPreRequirementCheckData()
    {
        return [
            [
                'requirement' => '<span data-toggle="tooltip" title="' . trans('fi.free_space') . '">Free Space</span>',
                'required'    => '100 MB',
                'actual'      => (getFreeSpace() > 100 ? getFreeSpace() . ' MB' : 'No'),
                'result'      => (getFreeSpace() > 100 ? 1 : 0),
            ],
            [
                'requirement' => 'PHP Version',
                'required'    => '8.1.0',
                'actual'      => PHP_VERSION,
                'result'      => ((version_compare(PHP_VERSION, '8.1.0') >= 0) ? 1 : 0),
            ],
            [
                'requirement' => 'Fileinfo Extension',
                'required'    => 'Yes',
                'actual'      => ((extension_loaded('fileinfo')) ? 'Yes' : 'No'),
                'result'      => ((extension_loaded('fileinfo')) ? 1 : 0),
            ],
            [
                'requirement' => 'OpenSSL Extension',
                'required'    => 'Yes',
                'actual'      => ((extension_loaded('openssl')) ? 'Yes' : 'No'),
                'result'      => ((extension_loaded('openssl')) ? 1 : 0),
            ],
            [
                'requirement' => 'PDO Extension',
                'required'    => 'Yes',
                'actual'      => ((extension_loaded('pdo')) ? 'Yes' : 'No'),
                'result'      => ((extension_loaded('pdo')) ? 1 : 0),
            ],
            [
                'requirement' => 'PDO MySQL Extension',
                'required'    => 'Yes',
                'actual'      => ((extension_loaded('pdo_mysql')) ? 'Yes' : 'No'),
                'result'      => ((extension_loaded('pdo_mysql')) ? 1 : 0),
            ],
            [
                'requirement' => 'MBString Extension',
                'required'    => 'Yes',
                'actual'      => ((extension_loaded('mbstring')) ? 'Yes' : 'No'),
                'result'      => ((extension_loaded('mbstring')) ? 1 : 0),
            ],
            [
                'requirement' => 'Tokenizer Extension',
                'required'    => 'Yes',
                'actual'      => ((extension_loaded('tokenizer')) ? 'Yes' : 'No'),
                'result'      => ((extension_loaded('tokenizer')) ? 1 : 0),
            ],
            [
                'requirement' => 'Graphics Drawing Extension',
                'required'    => 'Yes',
                'actual'      => ((extension_loaded('gd')) ? 'Yes' : 'No'),
                'result'      => ((extension_loaded('gd')) ? 1 : 0),
            ],
            [
                'requirement' => 'XML PHP Extension',
                'required'    => 'Yes',
                'actual'      => ((extension_loaded('xml')) ? 'Yes' : 'No'),
                'result'      => ((extension_loaded('xml')) ? 1 : 0),
            ],
            [
                'requirement' => 'DOM PHP Extension',
                'required'    => 'Yes',
                'actual'      => ((extension_loaded('dom')) ? 'Yes' : 'No'),
                'result'      => ((extension_loaded('dom')) ? 1 : 0),
            ],
            [
                'requirement' => 'Iconv PHP Extension',
                'required'    => 'Yes',
                'actual'      => ((extension_loaded('iconv')) ? 'Yes' : 'No'),
                'result'      => ((extension_loaded('iconv')) ? 1 : 0),
            ],
            [
                'requirement' => 'cURL PHP Extension',
                'required'    => 'Yes',
                'actual'      => ((extension_loaded('curl')) ? 'Yes' : 'No'),
                'result'      => ((extension_loaded('curl')) ? 1 : 0),
            ],
            [
                'requirement' => '<span data-toggle="tooltip" title="' . trans('fi.db_prefix_setup_alert') . '">Database Prefix</span>',
                'required'    => 'No',
                'actual'      => (DB::getTablePrefix() ? 'Yes' : 'No'),
                'result'      => (DB::getTablePrefix() ? 0 : 1),
            ],
        ];
    }

}