<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Sessions\Controllers;

use Exception;
use FI\Http\Controllers\Controller;
use FI\Jobs\PostLogin;
use FI\Jobs\SendOtpEmailJob;
use FI\Modules\CompanyProfiles\Models\CompanyProfile;
use FI\Modules\Sessions\Models\LoginOtp;
use FI\Modules\Sessions\Requests\LoginOtpRequest;
use FI\Modules\Sessions\Requests\SessionRequest;
use FI\Modules\Settings\Models\Setting;
use FI\Modules\Users\Models\User;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\File;

class SessionController extends Controller
{
    public function login()
    {
        if (!User::where('user_type', '<>', 'system')->count() || !Setting::where('setting_key', 'key')->count())
        {
            return redirect()->route('setup.postVerify.key');
        }

        $profileCode = request('profile', '');

        if ($profileCode)
        {
            try
            {
                $company_profile = CompanyProfile::whereUuid($profileCode)->where('logo', '!=', null)->select('logo')->first();
                $logo            = $company_profile != '' ? "data:image/png;base64," . base64_encode(file_get_contents(company_profile_logo_path($company_profile->logo))) : '';
            }
            catch (Exception $e)
            {
                $logo = '';
            }
        }
        else
        {
            try
            {
                $company_profile = CompanyProfile::whereIsDefault(1)->where('logo', '!=', null)->select('logo')->first();
                $logo            = $company_profile != '' ? "data:image/png;base64," . base64_encode(file_get_contents(company_profile_logo_path($company_profile->logo))) : '';
            }
            catch (Exception $e)
            {
                $logo = '';
            }
        }
        deleteTempFiles();
        deleteViewCache();

        if (config('fi.useCaptchaInLogin'))
        {
            generateCaptchaString();
        }
        return view('sessions.login')->with('logo', $logo)->with('databaseNames', databaseConnections());
    }

    public function attempt(SessionRequest $request)
    {

        $rememberMe = ($request->input('remember_me')) ? true : false;

        $user = User::whereEmail($request->input('email'))->where('user_type', '<>', 'system')->first();

        if (isset($user))
        {
            if (File::exists(storage_path('logs/laravel.log')))
            {
                $filePath   = storage_path('logs/laravel.log');
                $arrayBytes = ['UNIT' => 'MB', 'VALUE' => pow(1024, 2)];
                $fileSize   = strval(round(floatval(filesize($filePath)) / $arrayBytes['VALUE'], 0));

                if ($fileSize >= 5)
                {
                    $fp = fopen($filePath, 'a+');
                    ftruncate($fp, 1048576);
                    clearstatcache();
                    fclose($fp);
                }
            }

            if ($user->status == 0)
            {
                return redirect()->route('session.login')->with('alert', trans('fi.user_not_active'));
            }

            if ((!auth()->attempt(['email' => $request->input('email'), 'password' => $request->input('password'), 'status' => 1, 'user_type' => function ($query)
            {
                $query->where('user_type', '<>', 'paymentcenter_user');
            }], $rememberMe))
            )
            {
                return redirect()->route('session.login')->with('error', trans('fi.invalid_credentials'));
            }

            deleteCacheFiles();
            PostLogin::dispatch(auth()->user());

            if (auth()->user()->two_factor_enabled == 1)
            {
                try
                {
                    $otp      = LoginOtp::generateOtp();
                    $loginOtp = LoginOtp::updateOrCreate(['user_id' => auth()->user()->id], ['otp' => $otp, 'created_at' => now(), 'updated_at' => now()]);

                    $companyProfile = CompanyProfile::whereIsDefault('1')->first();

                    $data = ['loginOtp' => $loginOtp, 'email' => auth()->user()->email, 'ip' => $request->ip(), 'companyName' => $companyProfile->company ?? ''];

                    dispatch(new SendOtpEmailJob($data));

                    auth()->logout();

                    session([
                        'user_credentials' => [
                            'email' => $request->input('email'),
                        ]
                    ]);

                    return redirect()->route('session.otp-verification');
                }
                catch (Exception $e)
                {
                    auth()->logout();

                    return redirect()->route('session.login')->with('error', $e->getMessage());
                }
            }
            else
            {
                if (!auth()->user()->client_id)
                {
                    return redirect()->route('dashboard.index');
                }

                return redirect()->route('clientCenter.dashboard');
            }

        }
        else
        {
            return redirect()->route('session.login')->with('error', trans('fi.invalid_credentials'));
        }

    }

    public function logout()
    {
        if (config('time_tracking_enabled'))
        {
            event(new \Addons\TimeTracking\Events\StopTimeTrackerTasks(auth()->user()->id));
        }

        auth()->logout();

        session()->flush();

        Cookie::queue(Cookie::forget('time_tracking'));
        Cookie::queue(Cookie::forget('pricing_formula'));
        Cookie::queue(Cookie::forget('commission'));
        Cookie::queue(Cookie::forget('payment_center'));

        return redirect()->route('session.login');
    }

    public function refreshCaptcha()
    {
        generateCaptchaString();
        return "<img src='" . generateCaptchaImg() . "' width='100' alt='" . trans('fi.tt_captcha_image') . "'>";
    }

    public function checkMailServiceEnabled()
    {
        if (config('fi.mailDriver'))
        {
            return response()->json(['success' => true, 'message' => ''], 200);
        }
        else
        {
            return response()->json(['success' => false, 'message' => trans('fi.enabled_mail_driver')], 200);
        }
    }

    public function otpVerification()
    {
        $user_credentials = session('user_credentials');
        if (isset($user_credentials['email']))
        {

            try
            {
                $company_profile = CompanyProfile::whereIsDefault(1)->where('logo', '!=', null)->select('logo')->first();
                $logo            = $company_profile != '' ? "data:image/png;base64," . base64_encode(file_get_contents(company_profile_logo_path($company_profile->logo))) : '';
            }
            catch (Exception $e)
            {
                $logo = '';
            }
            return view('sessions.otp-verification')->with('logo', $logo);
        }
        else
        {
            return redirect()->route('session.login')->with('error', trans('fi.token_expired'));
        }
    }

    public function loginWithOtp(LoginOtpRequest $request)
    {
        $user_credentials = session('user_credentials');

        $user = User::with('loginOtp')->where('email', $user_credentials['email'])->where('user_type', '<>', 'system')->first();

        LoginOtp::where('user_id', $user->id)->delete();

        session(['user_credentials' => null]);

        auth()->login($user);

        return redirect()->route('dashboard.index');
    }

    public function resendOtp()
    {

        try
        {
            $user_credentials = session('user_credentials');

            $user = User::with('loginOtp')->where('email', $user_credentials['email'])->where('user_type', '<>', 'system')->first();

            $otp            = LoginOtp::generateOtp();
            $loginOtp       = LoginOtp::updateOrCreate(['user_id' => $user->id], ['otp' => $otp, 'created_at' => now(), 'updated_at' => now()]);
            $companyProfile = CompanyProfile::whereIsDefault('1')->first();

            $data = ['loginOtp' => $loginOtp, 'email' => $user->email, 'ip' => request()->ip(), 'companyName' => $companyProfile->company ?? ''];

            dispatch(new SendOtpEmailJob($data));

            return response()->json(['success' => true, 'message' => trans('fi.resend_otp_success')], 200);

        }
        catch (Exception $e)
        {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 200);
        }

    }
}