<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Quotes\Support;

use Carbon\Carbon;
use FI\Modules\CustomFields\Models\CustomField;
use FI\Modules\DocumentNumberSchemes\Models\DocumentNumberScheme;
use FI\Modules\Invoices\Events\InvoiceModified;
use FI\Modules\Invoices\Models\Invoice;
use FI\Modules\Invoices\Models\InvoiceItem;
use FI\Modules\PaymentTerms\Models\PaymentTerm;

class QuoteToInvoice
{
    public function convert($quote, $invoiceDate, $dueAt, $documentNumberSchemeId)
    {

        $record = [
            'invoice_date'                  => $invoiceDate,
            'status'                        => 'draft',
            'type'                          => 'invoice',
            'user_id'                       => $quote->user_id,
            'client_id'                     => $quote->client_id,
            'document_number_scheme_id'     => $documentNumberSchemeId,
            'due_at'                        => $dueAt,
            'number'                        => DocumentNumberScheme::generateNumber($documentNumberSchemeId, $quote->client->invoice_prefix),
            'terms'                         => ((config('fi.convertQuoteTerms') == 'quote') ? $quote->terms : config('fi.invoiceTerms')),
            'footer'                        => $quote->footer,
            'currency_code'                 => $quote->currency_code,
            'exchange_rate'                 => $quote->exchange_rate,
            'summary'                       => $quote->summary,
            'discount'                      => $quote->discount,
            'discount_type'                 => $quote->discount_type,
            'company_profile_id'            => $quote->company_profile_id,
            'payment_terms_id'              => $quote->payment_term_id,
            'online_payment_processing_fee' => $quote->online_payment_processing_fee,
        ];

        $quoteDate = Carbon::createFromDate($invoiceDate)->format('Y-m-d');
        if (!$record['payment_terms_id'])
        {
            $dueDays     = ($dueAt == null) ? 30 : Carbon::createFromDate($dueAt)->diffInDays($quoteDate);
            $newTerms    = (empty($quote->terms) == true) ? trans('fi.net_amount_due_invoice', ['days' => $dueDays]) : $quote->terms;
            $paymentTerm = PaymentTerm::where('net_due_days', $dueDays)->where('name', $newTerms)->first();
            if (empty($paymentTerm) == true)
            {
                $paymentTerm = PaymentTerm::create([
                    "type"         => 1,
                    "name"         => $newTerms,
                    "net_due_days" => $dueDays,
                ]);
            }
            $record['payment_terms_id'] = $paymentTerm->id;
        }

        $toInvoice                                = Invoice::create($record);
        $toInvoice->online_payment_processing_fee = $quote->online_payment_processing_fee;
        $toInvoice->save();

        CustomField::copyCustomFieldValues($quote, $toInvoice);

        $quote->invoice_id = $toInvoice->id;
        $quote->status     = 'approved';
        $quote->save();

        foreach ($quote->quoteItems as $item)
        {
            $itemRecord = [
                'invoice_id'    => $toInvoice->id,
                'name'          => $item->name,
                'description'   => $item->description,
                'quantity'      => $item->quantity,
                'price'         => $item->price,
                'tax_rate_id'   => $item->tax_rate_id,
                'tax_rate_2_id' => $item->tax_rate_2_id,
                'discount'      => $item->discount,
                'discount_type' => $item->discount_type,
                'display_order' => $item->display_order,
            ];

            $toInvoiceItem = InvoiceItem::create($itemRecord);
            CustomField::copyCustomFieldValues($item, $toInvoiceItem);

        }

        event(new InvoiceModified($toInvoice));

        return $toInvoice;
    }
}
