<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Payments;

use FI\Modules\Invoices\Models\Invoice;
use FI\Modules\MailQueue\Support\MailQueue;
use FI\Modules\Payments\Events\AddTransition;
use FI\Modules\Payments\Events\PaymentEmailed;
use FI\Modules\Payments\Events\PaymentInvoiceTransition;
use FI\Modules\Transitions\Models\Transitions;
use FI\Modules\Users\Models\User;
use FI\Support\Contacts;
use FI\Support\Parser;
use Illuminate\Support\Facades\Log;

class EventSubscriber
{
    public function getUserId($object)
    {
        return (isset(auth()->user()->id)) ? (auth()->user()->id) : ((isset($object->user_id) && $object->user_id != '') ? $object->user_id : User::where('name', 'System')->whereStatus(0)->first()->id);
    }

    public function addTransition(AddTransition $event)
    {
        try
        {
            $transition                      = new Transitions();
            $transition->user_id             = $this->getUserId($event->payment);
            $transition->client_id           = $event->payment->client->id;
            $transition->transitionable_id   = $event->payment->id;
            $transition->transitionable_type = 'FI\Modules\Payments\Models\Payment';
            $transition->action_type         = $event->actionType;
            if (!empty($event->detail))
            {
                $transition->detail = json_encode($event->detail);
            }
            $transition->previous_value = $event->previousValue;
            $transition->current_value  = $event->currentValue;
            $transition->save();
        }
        catch (\Exception $e)
        {
            Log::error('Payment transition entry failed: ' . $e->getMessage() . ' ' . $e->getFile() . ' ' . $e->getLine());
        }

    }

    public function paymentInvoiceTransition(PaymentInvoiceTransition $event)
    {

        $transition                      = new Transitions();
        $transition->user_id             = $this->getUserId($event->paymentInvoice->payment);
        $transition->client_id           = $event->paymentInvoice->invoice->client_id;
        $transition->transitionable_id   = $event->paymentInvoice->id;
        $transition->transitionable_type = 'FI\Modules\Payments\Models\PaymentInvoice';
        $transition->action_type         = $event->actionType;
        if (!empty($event->detail))
        {
            $transition->detail = json_encode($event->detail);
        }
        $transition->previous_value = $event->previousValue;
        $transition->current_value  = $event->currentValue;
        $transition->save();

    }

    public function paymentEmailed(PaymentEmailed $event)
    {
        // Change the status to sent if the status is currently draft
        $payment = $event->payment;
        $parser  = new Parser($payment);

        $contacts = new Contacts($payment->client);

        $mailQueue = new MailQueue();

        $invoice = Invoice::find($event->invoiceId);

        $template = !empty($invoice) ? $invoice->companyProfile->payment_receipt_email_body : $payment->client->companyProfile->payment_receipt_email_body;

        $mail = $mailQueue->create($payment, [
            'to'             => $contacts->getSelectedContactsTo(),
            'cc'             => $contacts->getSelectedContactsCc(),
            'bcc'            => $contacts->getSelectedContactsBcc(),
            'subject'        => $parser->companyProfileMailSubjectParse('paymentReceipt', $template),
            'body'           => $parser->companyProfileMailBodyParse('paymentReceipt', $template),
            'attach_pdf'     => 0,
            'attach_invoice' => config('fi.paymentAttachInvoice') ? config('fi.paymentAttachInvoice') : 0,
        ]);

        $mailQueue->send($mail->id);

        event(new AddTransition($payment, $event->actionType));

    }

    public function subscribe($events)
    {
        $events->listen('FI\Modules\Payments\Events\PaymentEmailed', 'FI\Modules\Payments\EventSubscriber@paymentEmailed');
        $events->listen('FI\Modules\Payments\Events\AddTransition', 'FI\Modules\Payments\EventSubscriber@addTransition');
        $events->listen('FI\Modules\Payments\Events\PaymentInvoiceTransition', 'FI\Modules\Payments\EventSubscriber@paymentInvoiceTransition');
    }
}
