<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\PaymentTerms\Models;

use Carbon\Carbon;
use FI\Support\CurrencyFormatter;
use FI\Support\NumberFormatter;
use FI\Traits\Sortable;
use Illuminate\Database\Eloquent\Model;

class PaymentTerm extends Model
{
    use Sortable;

    /**
     * Guarded properties
     * @var array
     */

    protected $table = 'payment_terms';

    protected $guarded = ['id'];

    protected $sortable = ['name', 'status'];

    /*
    |--------------------------------------------------------------------------
    | Relationships
    |--------------------------------------------------------------------------
    */

    public function installments()
    {
        return $this->hasMany('FI\Modules\PaymentTerms\Models\PaymentTermInstallment', 'payment_terms_id', 'id');
    }

    public function invoices()
    {
        return $this->hasMany('FI\Modules\Invoices\Models\Invoice', 'payment_terms_id', 'id');
    }

    public function quotes()
    {
        return $this->hasMany('FI\Modules\Quotes\Models\Quote', 'payment_term_id', 'id');
    }

    public function companyProfiles()
    {
        return $this->hasMany('FI\Modules\CompanyProfiles\Models\CompanyProfile', 'payment_term_id', 'id');
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors
    |--------------------------------------------------------------------------
    */

    public function getFormattedTypeAttribute()
    {
        return self::numberOfInstallments($this->attributes['type']);
    }

    public function getFormattedDiscountPercentAttribute()
    {
        return (in_array($this->attributes['type'], [2, 3])) ? (NumberFormatter::format(floatval($this->attributes['discount_percent'])) . ' %') : '';
    }

    public function getFormattedDiscountDaysSimpleAttribute()
    {
        return ($this->attributes['type'] == 2) ? $this->attributes['discount_days_simple'] . ' ' . trans('fi.days') : '';
    }

    public function getFormattedDiscountDaysNthAttribute()
    {
        return ($this->attributes['type'] == 3) ? trans('fi.day') . ' ' . $this->attributes['discount_days_nth'] . ' Next Month' : '';
    }

    public function getFormattedBalanceDueDaysAttribute()
    {
        return ($this->attributes['type'] == 4) ? $this->attributes['balance_due_days'] . ' ' . trans('fi.days') : '';
    }

    public function getFormattedNetDueDaysAttribute()
    {
        if (in_array($this->attributes['type'], [0, 1, 2, 3, 4]))
        {
            if ($this->attributes['net_due_days'] != '')
            {
                return $this->attributes['net_due_days'] . ' ' . trans('fi.days');
            }
        }
    }

    public function getFormattedNumberOfInstallmentsAttribute()
    {
        return ($this->attributes['type'] == 5) ? $this->attributes['number_of_installments'] . ' ' . trans('fi.installments') : '';
    }

    public function getFormattedActualTermsText($paymentTerm, $invoice = null)
    {
        return $this->typeNotice($paymentTerm, $invoice);
    }

    public function getNumericFormattedDiscountPercentAttribute()
    {
        return NumberFormatter::format($this->attributes['discount_percent']);
    }

    public function getNumericFormattedDownPaymentWithTypeAttribute()
    {
        if (!empty($this->attributes['down_payment']))
        {
            return NumberFormatter::format($this->attributes['down_payment']) . " <span class='badge badge-default'>" . trans('fi.' . $this->attributes['down_payment_type']) . "</span>";
        }

        return null;
    }

    public function getFormattedDownPaymentAttribute()
    {
        return NumberFormatter::format($this->attributes['down_payment']);
    }

    public function getFormattedStatusAttribute()
    {
        if ($this->status == 0)
        {
            return '<span class="badge badge-danger">' . trans('fi.inactive') . '</span>';
        }
        else
        {
            return '<span class="badge badge-success">' . trans('fi.active') . '</span>';
        }
    }

    public function getIsTypeLockedAttribute()
    {
        return ($this->invoices->isNotEmpty() || $this->quotes->isNotEmpty() || $this->companyProfiles->isNotEmpty());
    }

    /*
    |--------------------------------------------------------------------------
    | Static Functions
    |--------------------------------------------------------------------------
    */

    public static function numberOfInstallments($type = 0)
    {
        $types = [1 => trans('fi.one'), 2 => trans('fi.two'), 3 => trans('fi.three'), 4 => trans('fi.four'), 5 => trans('fi.five'), 6 => trans('fi.six')];
        if ($type != 0)
        {
            return $types[$type];
        }
        return $types;
    }

    public static function getDropDownList($addLabel = true)
    {
        if ($addLabel)
        {
            return ['' => trans('fi.select_payment_term')] + self::where('status', 1)->pluck('name', 'id')->all();
        }

        return self::where('status', 1)->pluck('name', 'id')->all();

    }

    public static function typeNotice($paymentTerm, $invoice)
    {
        switch ($paymentTerm->type)
        {
            case 0:
            case 1:
                return self::processTypeOne($paymentTerm, $invoice);
            case 2:
                return self::processTypeTwo($paymentTerm, $invoice);
            case 3:
                return self::processTypeThree($paymentTerm, $invoice);
            case 4:
                return self::processTypeFour($paymentTerm, $invoice);
            case 5:
                return self::processTypeFive($paymentTerm, $invoice);
            default:
                return '';
        }
    }

    private static function processTypeOne($paymentTerm, $invoice)
    {
        $message   = $paymentTerm->description_template;
        $nth_day   = self::ordinalNumber($paymentTerm->net_due_days ?? 0);
        $message   = self::modifyContent('#NTHDAY', $nth_day . ' ' . trans('fi.days'), $message);
        $last_date = Carbon::parse($invoice->invoice_date)->addDays($paymentTerm->net_due_days ?? 0)->format(config('fi.dateFormat'));
        $message   = self::modifyContent('#INVOICEDATE', Carbon::parse($invoice->invoice_date)->format(config('fi.dateFormat')), $message);
        return self::modifyContent('#LASTDATE', $last_date, $message);
    }

    private static function processTypeTwo($paymentTerm, $invoice)
    {
        $message = $paymentTerm->description_template;

        $discount_amount   = ($invoice->amount->total * ($paymentTerm->discount_percent / 100));
        $discounted_total  = $invoice->amount->total - $discount_amount;
        $no_discount_total = $invoice->amount->total;

        $discount_date = Carbon::parse($invoice->invoice_date)->addDays($paymentTerm->discount_days_simple ?? 0)->format(config('fi.dateFormat'));
        $last_date     = Carbon::parse($invoice->invoice_date)->addDays($paymentTerm->net_due_days ?? 0)->format(config('fi.dateFormat'));
        $message       = self::modifyContent('#DISCOUNTPERCENT', ($paymentTerm->numeric_formatted_discount_percent ?? 0), $message);

        $message = self::modifyContent('#INVOICEDATE', Carbon::parse($invoice->invoice_date)->format(config('fi.dateFormat')), $message);
        $message = self::modifyContent('#DISCOUNTAMOUNT', CurrencyFormatter::format($discount_amount), $message);
        $message = self::modifyContent('#DISCOUNTEDTOTAL', CurrencyFormatter::format($discounted_total), $message);
        $message = self::modifyContent('#NODISCOUNTTOTAL', CurrencyFormatter::format($no_discount_total), $message);
        $message = self::modifyContent('#DISCOUNTDATE', $discount_date, $message);
        return self::modifyContent('#LASTDATE', $last_date, $message);
    }

    private static function processTypeThree($paymentTerm, $invoice)
    {
        $message = $paymentTerm->description_template;

        $discount_amount   = ($invoice->amount->total * ($paymentTerm->discount_percent / 100));
        $discounted_total  = $invoice->amount->total - $discount_amount;
        $no_discount_total = $invoice->amount->total;

        $nth_day       = self::ordinalNumber($paymentTerm->discount_days_nth ?? 0);
        $discount_date = Carbon::parse($invoice->invoice_date)->endOfMonth()->addDays($paymentTerm->discount_days_nth ?? 0)->format(config('fi.dateFormat'));
        $last_date     = Carbon::parse($invoice->invoice_date)->addDays($paymentTerm->net_due_days ?? 0)->format(config('fi.dateFormat'));
        $message       = self::modifyContent('#DISCOUNTPERCENT', ($paymentTerm->numeric_formatted_discount_percent ?? 0), $message);
        $message       = self::modifyContent('#NTHDAY', $nth_day, $message);
        $message       = self::modifyContent('#DISCOUNTAMOUNT', CurrencyFormatter::format($discount_amount), $message);
        $message       = self::modifyContent('#DISCOUNTEDTOTAL', CurrencyFormatter::format($discounted_total), $message);
        $message       = self::modifyContent('#NODISCOUNTTOTAL', CurrencyFormatter::format($no_discount_total), $message);
        $message       = self::modifyContent('#INVOICEDATE', Carbon::parse($invoice->invoice_date)->format(config('fi.dateFormat')), $message);

        $message = self::modifyContent('#DISCOUNTDATE', $discount_date, $message);
        return self::modifyContent('#LASTDATE', $last_date, $message);
    }

    private static function processTypeFour($paymentTerm, $invoice)
    {
        $message             = $paymentTerm->description_template;
        $numeric_formatted   = number_format($paymentTerm->down_payment, 2, '.', '');
        $down_payment_amount = CurrencyFormatter::format($paymentTerm->down_payment ?? 0, $invoice->currency);
        $last_date           = Carbon::parse($invoice->invoice_date)->addDays($paymentTerm->balance_due_days ?? 0)->format(config('fi.dateFormat'));
        $message             = self::modifyContent('#INVOICEDATE', Carbon::parse($invoice->invoice_date)->format(config('fi.dateFormat')), $message);
        if ($paymentTerm->down_payment_type == 'flat_amount')
        {
            $message = self::modifyContent('#DOWNPAYMENT', $down_payment_amount, $message);
        }
        else
        {
            $message = self::modifyContent('#DOWNPAYMENTPERCENT', $numeric_formatted . '% ', $message);
        }

        return self::modifyContent('#LASTDATE', $last_date, $message);
    }

    private static function processTypeFive($paymentTerm, $invoice)
    {
        $full_message = $paymentTerm->description_template;
        $full_message = self::modifyContent('#INVOICEDATE', Carbon::parse($invoice->invoice_date)->format(config('fi.dateFormat')), $full_message);

        foreach ($paymentTerm->installments as $key => $installment)
        {
            $number = $key;

            $installmentNumber = $installment->installment_number ?? 0;

            $amount = ($invoice->amount->total * $installment->installment_percent / 100);

            $dateNew = Carbon::parse($invoice->invoice_date)->addDays($installment->installment_due_days ?? 0)->format(config('fi.dateFormat'));

            $message = str_replace('#NUMBEROFINSTALLMENTS', '<span class="text-info">' . ($paymentTerm->number_of_installments ?? 0) . '</span>', $full_message);

            $message = preg_replace('/#INSTALLMENTNUMBER/', '<span class="text-info">' . $installmentNumber . '</span>', $message, 1);

            $message = preg_replace('/#PERCENT/', '<span class="text-info">' . $installment->numeric_formatted_installment_percent . '</span>', $message, 1);

            $message = str_replace('#INSTALLMENTAMOUNT' . ($number + 1), '<span class="text-info">' . CurrencyFormatter::format(number_format(floor($amount * 100) / 100, 2, '.', ''), $invoice->currency) . '</span>', $message);

            $message = preg_replace('/#LASTDATE/', '<span class="text-info">' . $dateNew . '</span>', $message, 1);

            $full_message = $message;
        }

        return nl2br($full_message);
    }

    private static function ordinalNumber($number)
    {
        $suffix = ['th', 'st', 'nd', 'rd', 'th', 'th', 'th', 'th', 'th', 'th'];
        if (($number % 100) >= 11 && ($number % 100) <= 13)
        {
            $abbreviation = $number . 'th';
        }
        else
        {
            $abbreviation = $number . $suffix[$number % 10];
        }
        return $abbreviation;
    }

    private static function modifyContent($search, $replace, $string)
    {
        return str_replace($search, '<span class="text-info">' . $replace . '</span>', $string);
    }

    /*
   |--------------------------------------------------------------------------
   | Scopes
   |--------------------------------------------------------------------------
   */

    public function scopeKeywords($query, $keywords)
    {
        if ($keywords)
        {
            $keywords = strtolower($keywords);

            $query->where('payment_terms.type', 'like', '%' . $keywords . '%')
                ->orWhere('payment_terms.name', 'like', '%' . $keywords . '%')
                ->orWhere('payment_terms.discount_percent', 'like', '%' . $keywords . '%')
                ->orWhere('payment_terms.discount_days_simple', 'like', '%' . $keywords . '%')
                ->orWhere('payment_terms.discount_days_nth', 'like', '%' . $keywords . '%')
                ->orWhere('payment_terms.net_due_days', 'like', '%' . $keywords . '%');
        }

        return $query;
    }

    public function scopeDefaultEscape($query, $flag = true)
    {
        if ($flag)
        {
            return $query->where('is_default', '<>', 1);
        }
        return $query;
    }

}
