<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Office365\Support;

use Exception;
use FI\Modules\Settings\Models\Setting;
use Illuminate\Validation\ValidationException;
use Microsoft\Graph\Graph;
use Microsoft\Graph\Model\Message;

class Mail
{
    public $from;
    public $to;
    public $cc;
    public $bcc;
    public $subject;
    public $body;
    public $attachmentPath;
    public $invoiceAttachmentPath;

    public function __construct($from = null, $to = null, $cc = null, $bcc = null, $subject = null, $body = null, $attachmentPath = null, $invoiceAttachmentPath = null)
    {
        $this->from                  = $from;
        $this->to                    = $to;
        $this->cc                    = $cc;
        $this->bcc                   = $bcc;
        $this->subject               = $subject;
        $this->body                  = $body;
        $this->attachmentPath        = $attachmentPath;
        $this->invoiceAttachmentPath = $invoiceAttachmentPath;

    }

    public function sendTestEmail()
    {
        try
        {
            $graph     = new Graph();
            $token     = Setting::getByKey('microsoft_365_token');
            $userEmail = Setting::getByKey('office365Username');
            $graph->setAccessToken($token);

            $message = [
                'message' => [
                    'subject' => $this->subject,
                    'body'    => [
                        'contentType' => 'HTML',
                        'content'     => $this->body,
                    ],

                ],
            ];

            if ($this->to)
            {
                $toRecipients                       = $this->getRecipients($this->to);
                $message['message']['toRecipients'] = $toRecipients;
            }

            if ($this->cc)
            {
                $ccRecipients                       = $this->getRecipients($this->cc);
                $message['message']['ccRecipients'] = $ccRecipients;
            }

            if ($this->bcc)
            {
                $bccRecipients                       = $this->getRecipients($this->bcc);
                $message['message']['bccRecipients'] = $bccRecipients;
            }

            if (config('fi.mailReplyToAddress'))
            {
                $replyTo                       = $this->getReplyToEmail();
                $message['message']['replyTo'] = [$replyTo];
            }

            return $graph->createRequest('POST', '/users/' . $userEmail . '/sendMail')
                ->attachBody(new Message($message))
                ->execute();
        }
        catch (Exception $e)
        {
            throw ValidationException::withMessages([$e->getMessage()]);
        }

    }

    public function sendEmail()
    {
        try
        {
            $graph     = new Graph();
            $token     = Setting::getByKey('microsoft_365_token');
            $userEmail = Setting::getByKey('office365Username');
            $graph->setAccessToken($token);

            $message = [
                'message' => [
                    'subject' => $this->subject,
                    'body'    => [
                        'contentType' => 'HTML',
                        'content'     => $this->body,
                    ],

                ],
            ];

            if ($this->to)
            {
                $toRecipients                       = $this->getRecipients(json_decode($this->to));
                $message['message']['toRecipients'] = $toRecipients;
            }

            if ($this->cc)
            {
                $ccRecipients                       = $this->getRecipients(json_decode($this->cc));
                $message['message']['ccRecipients'] = $ccRecipients;
            }

            if ($this->bcc)
            {
                $bccRecipients                       = $this->getRecipients(json_decode($this->bcc));
                $message['message']['bccRecipients'] = $bccRecipients;
            }
            if (config('fi.mailReplyToAddress'))
            {

                $replyTo = $this->getReplyToEmail();

                $message['message']['replyTo'] = [$replyTo];
            }

            if ($this->attachmentPath)
            {
                $attachment                        = $this->getAttachment(basename(str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $this->attachmentPath)), $this->attachmentPath);
                $message['message']['attachments'] = [$attachment];
            }
            if ($this->invoiceAttachmentPath)
            {
                $invoiceAttachment                 = $this->getAttachment(basename(str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $this->attachmentPath)), $this->invoiceAttachmentPath);
                $message['message']['attachments'] = [$invoiceAttachment];

            }


            return $graph->createRequest('POST', '/users/' . $userEmail . '/sendMail')
                ->attachBody(new Message($message))
                ->execute();

        }
        catch (Exception $e)
        {
            throw ValidationException::withMessages([$e->getMessage()]);
        }

    }

    public function getAccessToken()
    {
        $url = 'https://login.microsoftonline.com/' . config('services.microsoft_graph.tenant_id') . '/oauth2/v2.0/token';

        $data = [
            'client_id'     => config('services.microsoft_graph.client_id'),
            'client_secret' => config('services.microsoft_graph.client_secret'),
            'scope'         => 'https://graph.microsoft.com/.default',
            'grant_type'    => 'client_credentials',
        ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

        $response      = curl_exec($ch);
        $response_data = json_decode($response, true);

        if (isset($response_data['access_token']))
        {
            return $response_data['access_token'];
        }
        else
        {
            throw ValidationException::withMessages(['Invalid microsoft graph API token']);
        }

    }

    public function getRecipients($emails)
    {

        $recipients = [];
        foreach ($emails as $email)
        {
            $recipients[] = [
                'emailAddress' => [
                    'address' => trim($email),
                ],
            ];

        }

        return $recipients;

    }

    public function getReplyToEmail()
    {
        $email = [
            'emailAddress' => [
                'address' => config('fi.mailReplyToAddress'),
            ],
        ];
        return $email;
    }

    public function getAttachment($name, $path)
    {
        $attachment = [
            '@odata.type'  => '#microsoft.graph.fileAttachment',
            'name'         => $name,
            'contentBytes' => base64_encode(file_get_contents($path)),
        ];

        return $attachment;
    }

}