<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\MerchantQuote\Support\Drivers;

use Carbon\Carbon;
use FI\Modules\Merchant\Models\MerchantPayment;
use FI\Modules\MerchantQuote\Support\MerchantDriver;
use FI\Modules\PaymentMethods\Models\PaymentMethod;
use FI\Modules\Payments\Events\PaymentEmailed;
use FI\Modules\Payments\Models\Payment;
use FI\Modules\Payments\Models\PaymentInvoice;
use FI\Modules\Quotes\Events\QuoteApproved;
use FI\Modules\Quotes\Events\QuoteToInvoiceTransition;
use FI\Modules\Quotes\Models\Quote;
use FI\Modules\Quotes\Support\QuoteToInvoice;
use FI\Modules\Users\Models\User;
use FI\Support\DateFormatter;
use Illuminate\Support\Facades\Log;
use Mollie\Api\MollieApiClient;

class MollieDriver extends MerchantDriver
{
    public function getSettings()
    {
        return ['apiKey'];
    }

    public function pay(Quote $quote)
    {
        try
        {
            $onlinePaymentChargesDetail = onlinePaymentQuoteChargesDetail($quote);
            $amountValue                = number_format($onlinePaymentChargesDetail['amountValue'], 2, '.', '');

            $mollie = new MollieApiClient();

            $mollie->setApiKey($this->getSetting('apiKey'));

            $payment = $mollie->payments->create(
                [
                    'amount'      =>
                        [
                            'currency' => $quote->currency_code,
                            'value'    => $amountValue,
                        ],
                    'description' => trans('fi.quote') . ' #' . $quote->number,
                    'redirectUrl' => route('merchant.pay.quote.mollie.return', [$quote->url_key]),
                    'webhookUrl'  => route('merchant.pay.quote.mollie.webhook', [$quote->url_key]),
                ]
            );

            return $payment->getCheckoutUrl();
        }
        catch (\Exception $e)
        {
            Log::error($e->getMessage());

            return redirect()->route('clientCenter.public.quote.show', [$quote->url_key, $quote->token])
                ->with('error', trans('fi.payment_problem'));
        }
    }

    public function verify(Quote $quote)
    {
        try
        {
            $onlinePaymentChargesDetail = onlinePaymentQuoteChargesDetail($quote);
            $paymentTermsDiscount       = $onlinePaymentChargesDetail['paymentTermsDiscount'];
            $mollieConvenienceCharges   = 0;

            if ($quote->online_payment_processing_fee == 'yes' && config('fi.feePercentage') != '')
            {
                $mollieConvenienceCharges = $onlinePaymentChargesDetail['feeCharges'];
            }

            $mollie = new MollieApiClient();

            $mollie->setApiKey($this->getSetting('apiKey'));

            $payment = $mollie->payments->get(request('id'));

            if ($payment->isPaid())
            {
                $quoteToInvoice = new QuoteToInvoice();

                $invoice = $quoteToInvoice->convert(
                    $quote,
                    date('Y-m-d'),
                    DateFormatter::incrementDateByDays(date('Y-m-d'), invoicesDueAfter()),
                    config('fi.invoiceGroup')
                );

                $userId = User::whereUserType('system')->first()->id;

                event(new QuoteToInvoiceTransition($quote, $invoice, $userId));

                onlineConvenienceCharges($invoice, $mollieConvenienceCharges);

                if ($paymentTermsDiscount != 0)
                {
                    $invoice->amount->payment_terms_discount += $paymentTermsDiscount;
                    $invoice->amount->save();
                }

                $userId              = (auth()->user() != null) ? ((auth()->user()->user_type == 'paymentcenter_user') ? auth()->user()->id : User::whereUserType('system')->first()->id) : User::whereUserType('system')->first()->id;
                $onlinePaymentMethod = PaymentMethod::whereName(trans('fi.online_payment'))->first();

                $fiPayment = Payment::create([
                    'user_id'           => $userId,
                    'client_id'         => $invoice->client->id,
                    'amount'            => $payment->amount->value,
                    'currency_code'     => $invoice->currency_code,
                    'remaining_balance' => 0,
                    'payment_method_id' => ($onlinePaymentMethod != null) ? $onlinePaymentMethod->id : config('fi.onlinePaymentMethod'),
                    'paid_at'           => Carbon::now()->format('Y-m-d'),
                ]);

                if ($fiPayment)
                {
                    $paymentInvoice                         = new PaymentInvoice();
                    $paymentInvoice->payment_id             = $fiPayment->id;
                    $paymentInvoice->invoice_id             = $invoice->id;
                    $paymentInvoice->invoice_amount_paid    = $payment->amount->value;
                    $paymentInvoice->convenience_charges    = $mollieConvenienceCharges;
                    $paymentInvoice->payment_terms_discount = $paymentTermsDiscount;
                    $paymentInvoice->installment_number     = $onlinePaymentChargesDetail['installment'];
                    $paymentInvoice->save();

                    event(new QuoteApproved($quote, 'quote_payment_successfully'));

                    if ($fiPayment->client->should_email_payment_receipt)
                    {
                        event(new PaymentEmailed($fiPayment));
                    }
                }
                if (isset($fiPayment->id) && isset($payment->id))
                {
                    MerchantPayment::saveByKey($this->getName(), $fiPayment->id, 'id', $payment->id);
                }
            }

        }
        catch (\Exception $e)
        {
            Log::error($e->getMessage());

            return redirect()->route('clientCenter.public.quote.show', [$quote->url_key, $quote->token])
                ->with('error', trans('fi.payment_problem'));
        }
    }
}