<?php

/**
 * This file is part of FusionInvoice.
 *
 * ( c ) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Merchant\Support\Drivers;

use Carbon\Carbon;
use FI\Modules\Invoices\Models\Invoice;
use FI\Modules\Merchant\Models\MerchantPayment;
use FI\Modules\Merchant\Support\MerchantDriver;
use FI\Modules\PaymentMethods\Models\PaymentMethod;
use FI\Modules\Payments\Events\PaymentEmailed;
use FI\Modules\Payments\Models\Payment;
use FI\Modules\Payments\Models\PaymentInvoice;
use FI\Modules\Quotes\Events\QuoteApproved;
use FI\Modules\Quotes\Models\Quote;
use FI\Modules\Users\Models\User;
use FI\Support\CurrencyFormatter;
use Illuminate\Support\Facades\Log;
use Square\Models\CreatePaymentRequest;
use Square\Models\Money;
use Square\SquareClient;

class SquareDriver extends MerchantDriver
{
    public function getSettings()
    {
        return ['accessToken', 'applicationId', 'mode' => ['sandbox' => trans('fi.sandbox'), 'production' => trans('fi.production')]];
    }

    private function getApiContext()
    {
        $square = new SquareClient([
            'accessToken' => $this->getSetting('accessToken'),
            'environment' => $this->getSetting('mode'),
        ]);

        return $square;
    }

    public function pay(Invoice $invoice, $modal, $quoteUrlKey)
    {
        Log::info('Pay redirect on merchant page');
        $onlinePaymentChargesDetail = onlinePaymentChargesDetail($invoice);
        $payableAmount              = CurrencyFormatter::format(($onlinePaymentChargesDetail['amountValue'] + 0), $invoice->currency);
        return view('merchant.square')
            ->with('appId', $this->getSetting('applicationId'))
            ->with('invoiceToken', $invoice->url_key)
            ->with('payableAmount', $payableAmount)
            ->with('quoteUrlKey', $quoteUrlKey)
            ->with('modal', $modal)
            ->with('environment', $this->getSetting('mode'));
    }

    public function success(Invoice $invoice, $modal, $quoteUrlKey, $sourceId)
    {
        try
        {

            $onlinePaymentChargesDetail = onlinePaymentChargesDetail($invoice);
            $paymentTermsDiscount       = $onlinePaymentChargesDetail['paymentTermsDiscount'];

            $amountValue     = floatval($onlinePaymentChargesDetail['amountValue']) * 100;
            $square          = $this->getApiContext();
            $paymentsApi     = $square->getPaymentsApi();
            $idempotency_key = uniqid();

            $body_amountMoney = new Money;
            $body_amountMoney->setAmount($amountValue);
            $body_amountMoney->setCurrency($invoice->currency_code);
            $body = new CreatePaymentRequest($sourceId, $idempotency_key);

            $body->setAmountMoney($body_amountMoney);
            $body->setAutocomplete(true);
            $body->setNote(trans('fi.invoice') . ' #' . $invoice->number);

            $result = $paymentsApi->createPayment($body);

            if (!$result->isSuccess())
            {
                return response()->json(
                    [
                        'success'     => false,
                        'redirectUrl' => route('clientCenter.public.invoice.show', [$invoice->url_key, $invoice->token]),
                        'message'     => trans('fi.payment_problem')
                    ], 400);
            }
            else
            {
                $payment             = (array)json_decode($result->getBody(), true);
                $approved_money      = ($payment['payment']['approved_money']['amount']);
                $approved_payment_id = ($payment['payment']['id']);

                $squareConvenienceCharges = 0;

                if ($invoice->online_payment_processing_fee == 'yes' && config('fi.feePercentage') != '')
                {
                    $squareConvenienceCharges = $onlinePaymentChargesDetail['feeCharges'];
                }

                onlineConvenienceCharges($invoice, $squareConvenienceCharges);

                if ($paymentTermsDiscount != 0)
                {
                    $invoice->amount->payment_terms_discount += $paymentTermsDiscount;
                    $invoice->amount->save();
                }

                $userId              = (auth()->user() != null) ? ((auth()->user()->user_type == 'paymentcenter_user') ? auth()->user()->id : User::whereUserType('system')->first()->id) : User::whereUserType('system')->first()->id;
                $onlinePaymentMethod = PaymentMethod::whereName(trans('fi.online_payment'))->first();

                $fiPayment = Payment::create([
                    'user_id'           => $userId,
                    'client_id'         => $invoice->client->id,
                    'amount'            => $approved_money / 100,
                    'currency_code'     => $invoice->currency_code,
                    'remaining_balance' => 0,
                    'payment_method_id' => ($onlinePaymentMethod != null) ? $onlinePaymentMethod->id : config('fi.onlinePaymentMethod'),
                    'paid_at'           => Carbon::now()->format('Y-m-d'),
                ]);

                if ($fiPayment)
                {
                    $paymentInvoice                         = new PaymentInvoice();
                    $paymentInvoice->payment_id             = $fiPayment->id;
                    $paymentInvoice->invoice_id             = $invoice->id;
                    $paymentInvoice->invoice_amount_paid    = $approved_money / 100;
                    $paymentInvoice->convenience_charges    = $squareConvenienceCharges;
                    $paymentInvoice->payment_terms_discount = $paymentTermsDiscount;
                    $paymentInvoice->installment_number     = $onlinePaymentChargesDetail['installment'];
                    $paymentInvoice->save();

                    if (!empty($quoteUrlKey) && $quoteUrlKey != 'fi')
                    {
                        $quote = Quote::whereUrlKey($quoteUrlKey)->first();
                        if (isset($quote->id))
                        {
                            event(new QuoteApproved($quote, 'quote_payment_successfully'));
                        }
                    }

                    if ($fiPayment->client->should_email_payment_receipt)
                    {
                        event(new PaymentEmailed($fiPayment));
                    }
                }
                if (isset($fiPayment->id) && $approved_payment_id != '')
                {
                    MerchantPayment::saveByKey('Square', $fiPayment->id, 'id', $approved_payment_id);
                }

                return response()->json(
                    ['success'     => true,
                     'redirectUrl' => route('clientCenter.public.invoice.show', [$invoice->url_key, $invoice->token]),
                     'message'     => trans('fi.payment_applied')
                    ], 200);

            }
        }
        catch (\Exception $e)
        {
            if ($quoteUrlKey != '')
            {
                if ($modal == 'pi' && class_exists(\Addons\ProformaInvoice\Models\ProformaInvoice::class) && config('proforma_invoice_enabled') == true)
                {
                    $proformaInvoice = \Addons\ProformaInvoice\Models\ProformaInvoice::where('url_key', $quoteUrlKey)->first();
                    $proformaInvoice->invoice->delete();
                    return response()->json([
                        'success'     => false,
                        'redirectUrl' => route('clientCenter.public.proformaInvoice.show', [$quoteUrlKey, $proformaInvoice->token]),
                        'message'     => trans('fi.payment_failed', ['error' => $e->getMessage()])
                    ], 401);
                }
                else
                {
                    $quote = Quote::where('url_key', $quoteUrlKey)->first();
                    $quote->invoice->delete();
                    return response()->json([
                        'success'     => false,
                        'redirectUrl' => route('clientCenter.public.quote.show', [$quoteUrlKey, $quote->token]),
                        'message'     => trans('fi.payment_failed', ['error' => $e->getMessage()])
                    ], 401);
                }

            }

            return response()->json([
                'success'     => false,
                'redirectUrl' => route('clientCenter.public.invoice.show', [$invoice->url_key, $invoice->token]),
                'message'     => trans('fi.payment_failed', ['error' => $e->getMessage()])
            ], 401);
        }

    }

    public function cancel($urlKey, $token = null, $modal = 'invoice', $quoteUrlKey = null)
    {
        if ($quoteUrlKey != '')
        {
            if ($modal == 'pi' && class_exists(\Addons\ProformaInvoice\Models\ProformaInvoice::class) && config('proforma_invoice_enabled') == true)
            {
                $proformaInvoice = \Addons\ProformaInvoice\Models\ProformaInvoice::where('url_key', $quoteUrlKey)->first();
                $proformaInvoice->invoice->delete();
                return redirect()->route('clientCenter.public.proformaInvoice.show', [$quoteUrlKey, $proformaInvoice->token])
                    ->with('alert', trans('fi.payment_problem'));
            }
            else
            {

                $quote = Quote::where('url_key', $quoteUrlKey)->first();
                $quote->invoice->delete();
                return redirect()->route('clientCenter.public.quote.show', [$quoteUrlKey, $quote->token])
                    ->with('alert', trans('fi.payment_canceled'));
            }

        }

        return redirect()->route('clientCenter.public.invoice.show', [$urlKey, $token])
            ->with('alert', trans('fi.payment_canceled'));
    }
}