<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Merchant\Support\Drivers;

use Carbon\Carbon;
use FI\Modules\Invoices\Models\Invoice;
use FI\Modules\Merchant\Models\MerchantPayment;
use FI\Modules\Merchant\Support\MerchantDriver;
use FI\Modules\PaymentMethods\Models\PaymentMethod;
use FI\Modules\Payments\Events\PaymentEmailed;
use FI\Modules\Payments\Models\Payment as FIPayment;
use FI\Modules\Payments\Models\PaymentInvoice;
use FI\Modules\Quotes\Events\QuoteApproved;
use FI\Modules\Quotes\Models\Quote;
use FI\Modules\Users\Models\User;
use Illuminate\Support\Facades\Log;
use Srmklive\PayPal\Services\PayPal as PayPalClient;


class PayPalDriver extends MerchantDriver
{
    public function getSettings()
    {
        return ['clientId', 'clientSecret', 'mode' => ['sandbox' => trans('fi.sandbox'), 'live' => trans('fi.live')]];
    }

    public function pay(Invoice $invoice, $modal, $quoteUrlKey)
    {
        $onlinePaymentChargesDetail = onlinePaymentChargesDetail($invoice);
        $amountValue                = $onlinePaymentChargesDetail['amountValue'];
        $addChargesItem             = $onlinePaymentChargesDetail['addChargesItem'];
        $feeCharges                 = $onlinePaymentChargesDetail['feeCharges'];
        $originalAmountValue        = $onlinePaymentChargesDetail['originalAmountValue'];

        $provider = $this->getApiContext();

        $data = [
            "intent"         => "CAPTURE",
            "purchase_units" => [
                [
                    "reference_id" => $invoice->number,
                    "items"        => [
                        [
                            "name"        => trans('fi.invoice') . ' #' . $invoice->number,
                            "quantity"    => "1",
                            "unit_amount" => [
                                "currency_code" => $invoice->currency_code,
                                "value"         => (number_format($originalAmountValue, 2, '.', '')),
                            ]
                        ],
                        [
                            "name"        => trans('fi.convenience_charges'),
                            "quantity"    => "1",
                            "unit_amount" => [
                                "currency_code" => $invoice->currency_code,
                                "value"         => ($addChargesItem == true) ? (number_format($feeCharges, 2, '.', '')) : '0',
                            ]
                        ],
                    ],
                    "amount"       => [
                        "currency_code" => $invoice->currency_code,
                        "value"         => (number_format($amountValue, 2, '.', '')),
                        "breakdown"     => [
                            "item_total" => [
                                "currency_code" => $invoice->currency_code,
                                "value"         => (number_format($amountValue, 2, '.', '')),
                            ]
                        ]
                    ]
                ]
            ],
            "payment_source" => [
                "paypal" => [
                    "experience_context" => [
                        "payment_method_preference" => "IMMEDIATE_PAYMENT_REQUIRED",
                        "locale"                    => "en-US",
                        "landing_page"              => "LOGIN",
                        "user_action"               => "PAY_NOW",
                        "return_url"                => route('merchant.pay.paypal.return', [$invoice->url_key, $modal, $quoteUrlKey]),
                        "cancel_url"                => route('merchant.pay.paypal.cancel', [$invoice->url_key, $modal, $quoteUrlKey]),
                    ]
                ]
            ]
        ];

        try
        {

            $order = $provider->createOrder($data);
            if (isset($order['error']))
            {
                Log::error($order['error']['message']);

                if ($quoteUrlKey != '')
                {
                    $quote = Quote::where('url_key', $quoteUrlKey)->first();
                    $quote->invoice->delete();
                    return redirect()->route('clientCenter.public.quote.show', [$quoteUrlKey, $quote->token])
                        ->with('error', $order['error']['message']);
                }

                if ($quoteUrlKey != '')
                {
                    if ($modal == 'pi' && class_exists(\Addons\ProformaInvoice\Models\ProformaInvoice::class) && config('proforma_invoice_enabled') == true)
                    {
                        $proformaInvoice = \Addons\ProformaInvoice\Models\ProformaInvoice::where('url_key', $quoteUrlKey)->first();
                        $proformaInvoice->invoice->delete();
                        return redirect()->route('clientCenter.public.proformaInvoice.show', [$quoteUrlKey, $proformaInvoice->token])
                            ->with('error', $order['error']['message']);
                    }
                    else
                    {
                        $quote = Quote::where('url_key', $quoteUrlKey)->first();
                        $quote->invoice->delete();
                        return redirect()->route('clientCenter.public.quote.show', [$quoteUrlKey, $quote->token])
                            ->with('error', $order['error']['message']);
                    }
                }


                return redirect()->route('clientCenter.public.invoice.show', [$invoice->url_key, $invoice->token])
                    ->with('error', $order['error']['message']);
            }
            else
            {
                return redirect($order['links'][1]['href']);
            }

        }
        catch (\Exception $e)
        {
            Log::error($e->getMessage());

            if ($quoteUrlKey != '')
            {
                $quote = Quote::where('url_key', $quoteUrlKey)->first();
                $quote->invoice->delete();
                return redirect()->route('clientCenter.public.quote.show', [$quoteUrlKey, $quote->token])
                    ->with('alert', trans('fi.payment_problem'));
            }

            return redirect()->route('clientCenter.public.invoice.show', [$invoice->url_key, $invoice->token])
                ->with('error', trans('fi.payment_problem'));
        }
    }

    public function success(Invoice $invoice, $modal, $quoteUrlKey)
    {
        try
        {
            $onlinePaymentChargesDetail = onlinePaymentChargesDetail($invoice);

            $paymentTermsDiscount = $onlinePaymentChargesDetail['paymentTermsDiscount'];

            $paymentConvenienceCharges = 0;

            if ($invoice->online_payment_processing_fee == 'yes' && config('fi.feePercentage') != '')
            {
                $paymentConvenienceCharges = $onlinePaymentChargesDetail['feeCharges'];
            }

            $provider = $this->getApiContext();
            $payment  = $provider->capturePaymentOrder(request('token'));

            if (isset($payment['status']) && $payment['status'] == 'COMPLETED')
            {
                try

                {
                    $paymentId    = $payment['id'];
                    $transactions = $payment['purchase_units'][0]['payments']['captures'];
                    onlineConvenienceCharges($invoice, $paymentConvenienceCharges);

                    if ($paymentTermsDiscount != 0)
                    {
                        $invoice->amount->payment_terms_discount += $paymentTermsDiscount;
                        $invoice->amount->save();
                    }

                    $userId              = (auth()->user() != null) ? ((auth()->user()->user_type == 'paymentcenter_user') ? auth()->user()->id : User::whereUserType('system')->first()->id) : User::whereUserType('system')->first()->id;
                    $onlinePaymentMethod = PaymentMethod::whereName(trans('fi.online_payment'))->first();

                    foreach ($transactions as $transaction)
                    {
                        $fiPayment = FIPayment::create([
                            'user_id'           => $userId,
                            'client_id'         => $invoice->client->id,
                            'amount'            => $transaction['amount']['value'],
                            'currency_code'     => $invoice->currency_code,
                            'remaining_balance' => 0,
                            'payment_method_id' => ($onlinePaymentMethod != null) ? $onlinePaymentMethod->id : config('fi.onlinePaymentMethod'),
                            'paid_at'           => Carbon::now()->format('Y-m-d'),
                        ]);
                        if ($fiPayment)
                        {
                            $paymentInvoice                         = new PaymentInvoice();
                            $paymentInvoice->payment_id             = $fiPayment->id;
                            $paymentInvoice->invoice_id             = $invoice->id;
                            $paymentInvoice->invoice_amount_paid    = $transaction['amount']['value'];
                            $paymentInvoice->convenience_charges    = $paymentConvenienceCharges;
                            $paymentInvoice->payment_terms_discount = $paymentTermsDiscount;
                            $paymentInvoice->installment_number     = $onlinePaymentChargesDetail['installment'];
                            $paymentInvoice->save();

                            if (!empty($quoteUrlKey) && $quoteUrlKey != 'fi')
                            {
                                $quote = Quote::whereUrlKey($quoteUrlKey)->first();
                                if (isset($quote->id))
                                {
                                    event(new QuoteApproved($quote, 'quote_payment_successfully'));
                                }
                            }

                            if ($fiPayment->client->should_email_payment_receipt)
                            {
                                event(new PaymentEmailed($fiPayment));
                            }
                        }
                        if (isset($fiPayment->id) && $paymentId != '')
                        {
                            MerchantPayment::saveByKey('PayPal', $fiPayment->id, 'id', $paymentId);
                        }
                    }
                    return redirect()->route('clientCenter.public.invoice.show', [$invoice->url_key, $invoice->token])
                        ->with('alertSuccess', trans('fi.payment_applied'));

                }
                catch (\Exception $e)
                {
                    Log::error($e->getFile() . ' ' . $e->getMessage());
                }
            }

            if ($quoteUrlKey != '')
            {
                if ($modal == 'pi' && class_exists(\Addons\ProformaInvoice\Models\ProformaInvoice::class) && config('proforma_invoice_enabled') == true)
                {
                    $proformaInvoice = \Addons\ProformaInvoice\Models\ProformaInvoice::where('url_key', $quoteUrlKey)->first();
                    $proformaInvoice->invoice->delete();
                    return redirect()->route('clientCenter.public.proformaInvoice.show', [$quoteUrlKey, $proformaInvoice->token])
                        ->with('alert', trans('fi.payment_problem'));
                }
                else
                {
                    $quote = Quote::where('url_key', $quoteUrlKey)->first();
                    $quote->invoice->delete();
                    return redirect()->route('clientCenter.public.quote.show', [$quoteUrlKey, $quote->token])
                        ->with('alert', trans('fi.payment_problem'));
                }
            }

            return redirect()->route('clientCenter.public.invoice.show', [$invoice->url_key, $invoice->token])
                ->with('alert', trans('fi.payment_problem'));

        }
        catch (\Exception $e)
        {
            Log::error($e->getFile() . ' ' . $e->getMessage());
        }

    }

    private function getApiContext()
    {
        $config = [
            'mode'           => $this->getSetting('mode'),
            'sandbox'        => [
                'client_id'     => $this->getSetting('clientId'),
                'client_secret' => $this->getSetting('clientSecret'),
            ],
            'live'           => [
                'client_id'     => $this->getSetting('clientId'),
                'client_secret' => $this->getSetting('clientSecret'),
            ],
            'payment_action' => 'Sale',
            'currency'       => config('fi.baseCurrency'),
            'locale'         => 'en_US',
            'validate_ssl'   => true,
            'notify_url'     => url('/'),
        ];

        $provider = new PayPalClient($config);
        $provider->getAccessToken();

        return $provider;
    }

    public function cancel($urlKey, $token = null, $modal = 'invoice', $quoteUrlKey = null)
    {
        if ($quoteUrlKey != '')
        {
            if ($modal == 'pi' && class_exists(\Addons\ProformaInvoice\Models\ProformaInvoice::class) && config('proforma_invoice_enabled') == true)
            {
                $proformaInvoice = \Addons\ProformaInvoice\Models\ProformaInvoice::where('url_key', $quoteUrlKey)->first();
                $proformaInvoice->invoice->delete();
                return redirect()->route('clientCenter.public.proformaInvoice.show', [$quoteUrlKey, $proformaInvoice->token])
                    ->with('alert', trans('fi.payment_canceled'));
            }
            else
            {
                $quote = Quote::where('url_key', $quoteUrlKey)->first();
                $quote->invoice->delete();
                return redirect()->route('clientCenter.public.quote.show', [$quoteUrlKey, $quote->token])
                    ->with('alert', trans('fi.payment_canceled'));
            }
        }

        return redirect()->route('clientCenter.public.invoice.show', [$urlKey, $token])
            ->with('alert', trans('fi.payment_canceled'));
    }
}
