<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\MailQueue\Controllers;

use Addons\ProformaInvoice\Events\AddTransition as ProformaTransition;
use Addons\ProformaInvoice\Events\ProformaInvoiceEmailed;
use Addons\ProformaInvoice\Models\ProformaInvoice;
use FI\Http\Controllers\Controller;
use FI\Modules\Invoices\Events\AddTransition as InvoiceTransition;
use FI\Modules\Invoices\Events\InvoiceEmailed;
use FI\Modules\Invoices\Models\Invoice;
use FI\Modules\MailQueue\Models\MailQueue;
use FI\Modules\MailQueue\Support\MailQueue as SupportMail;
use FI\Modules\Payments\Events\PaymentEmailed;
use FI\Modules\Payments\Models\Payment;
use FI\Modules\Quotes\Events\AddTransition as QuoteTransition;
use FI\Modules\Quotes\Events\QuoteEmailed;
use FI\Modules\Quotes\Models\Quote;

class MailLogController extends Controller
{
    public function index()
    {
        if (!config('app.demo'))
        {
            $mails = MailQueue::sortable(['created_at' => 'desc'])
                ->keywords(request('search'))
                ->paginate(config('fi.resultsPerPage'));

            return view('mail_log.index')
                ->with('mails', $mails)
                ->with('searchPlaceholder', trans('fi.search_log'));
        }
        else
        {
            return redirect()->route('dashboard.index')
                ->withErrors(trans('fi.functionality_not_available_on_demo'));
        }
    }

    public function content()
    {
        $mail = MailQueue::select('subject', 'body')
            ->where('id', request('id'))
            ->first();

        return view('mail_log._modal_content')
            ->with('mail', $mail);
    }

    public function delete($id)
    {
        MailQueue::destroy($id);

        return redirect()->route('mailLog.index')
            ->with('alert', trans('fi.record_successfully_deleted'));
    }

    public function deleteModal()
    {
        try
        {
            return view('layouts._delete_modal_details')->with('url', request('action'))->with('returnURL', request('returnURL'))->with('modalName', request('modalName'))->with('isReload', request('isReload'));
        }
        catch (\Exception $e)
        {
            return response()->json(['success' => false, 'message' => ($e->getCode() == 0) ? trans('fi.modal_not_found') : $e->getMessage()], 400);
        }
    }

    public function resendModal()
    {
        try
        {
            return view('mail_log._resend_email_modal')->with('url', request('action'))->with('returnURL', request('returnURL'))->with('modalName', request('modalName'))->with('isReload', request('isReload'));
        }
        catch (\Exception $e)
        {
            return response()->json(['success' => false, 'message' => ($e->getCode() == 0) ? trans('fi.modal_not_found') : $e->getMessage()], 400);
        }
    }

    public function resend($id)
    {
        try
        {
            $mailQueue = MailQueue::find($id);
            $support   = new SupportMail();

            if ($mailQueue->mailable_type == Payment::class)
            {
                event(new PaymentEmailed($mailQueue->mailable, 'payment_receipt_resend_email'));
            }
            elseif ($support->send($id))
            {
                switch ($mailQueue->mailable_type)
                {
                    case Invoice::class:
                        event(new InvoiceEmailed($mailQueue->mailable));
                        event(new InvoiceTransition($mailQueue->mailable, 'resend_email'));
                        break;
                    case Quote::class:
                        event(new QuoteEmailed($mailQueue->mailable));
                        event(new QuoteTransition($mailQueue->mailable, 'resend_email'));
                        break;
                    case ProformaInvoice::class:
                        event(new ProformaInvoiceEmailed($mailQueue->mailable));
                        event(new ProformaTransition($mailQueue->mailable, 'resend_email'));
                        break;
                    default:
                        break;
                }
            }
            else
            {
                return response()->json(['errors' => [[$mailQueue->getError()]]], 400);
            }

            return response()->json(['message' => trans('fi.resend_email')], 200);
        }
        catch (\Exception $e)
        {
            return response()->json(['success' => false, 'message' => ($e->getCode() == 0) ? trans('fi.modal_not_found') : $e->getMessage()], 400);
        }
    }
}