<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Import\Importers;

use FI\Modules\Clients\Models\Client;
use FI\Modules\CompanyProfiles\Models\CompanyProfile;
use FI\Modules\CustomFields\Models\CustomField;
use FI\Modules\Users\Models\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class ClientImporter extends AbstractImporter
{
    public function getFields()
    {
        $fields = [
            'name'               => '* ' . trans('fi.name'),
            'type'               => trans('fi.type'),
            'address'            => trans('fi.address'),
            'address2'           => trans('fi.address2'),
            'city'               => trans('fi.city'),
            'state'              => trans('fi.state'),
            'zip'                => trans('fi.postal_code'),
            'country'            => trans('fi.country'),
            'phone'              => trans('fi.phone'),
            'fax'                => trans('fi.fax'),
            'mobile'             => trans('fi.mobile'),
            'email'              => trans('fi.email'),
            'web'                => trans('fi.web'),
            'company_profile_id' => trans('fi.company_profile'),
        ];

        foreach (CustomField::forTable('clients')->get() as $customField)
        {
            $fields['custom_' . $customField->column_name] = $customField->field_label;
        }

        return $fields;
    }

    public function getMapRules()
    {
        return [
            'name' => 'required',
        ];
    }

    public function getValidator($input)
    {
        return Validator::make($input, [
            'name'               => 'required',
            'email'              => 'email',
            'type'               => 'required|in:lead,prospect,customer,affiliate,other',
            'company_profile_id' => 'required|numeric',
        ])->setAttributeNames(['name' => trans('fi.name')]);
    }

    public function importData($input)
    {
        $this->file   = storage_path('clients.csv');
        $row          = 1;
        $fields       = $customFields = [];
        $response     = ['success' => false, 'total_records' => 0, 'message' => ''];
        $cmpProfileId = CompanyProfile::where('is_default', 1)->first()->id;

        try
        {
            foreach ($input as $key => $field)
            {
                if ($field != '')
                {
                    if (substr($key, 0, 7) != 'custom_')
                    {
                        $fields[$key] = $field;
                    }
                    else
                    {
                        $customFields[substr($key, 7)] = $field;
                    }
                }
            }

            $handle = fopen($this->file, 'r');

            if (!$handle)
            {
                $this->messages->add('error', 'Could not open the file');

                return $response;
            }

            $validationResponse = $this->validateTotalRecords($this->file);

            if ($validationResponse !== 'ok')
            {
                $response['message'] = $validationResponse;
                $this->messages->add('error', $response['message']);
                return $response;
            }

            $userId = User::whereUserType('system')->first()->id;

            DB::transaction(function () use (&$handle, &$fields, &$response, &$customFields, &$row, &$userId ,&$cmpProfileId)
            {
                while (($data = fgetcsv($handle, $this->max_records, ',')) !== false)
                {
                    if ($row !== 1 && $data !== [null])
                    {
                        $record = $customRecord = [];

                        foreach ($fields as $field => $key)
                        {
                            $record[$field] = trim($data[$key]);
                        }

                        if (isset($record['address2']))
                        {
                            $record['address'] = $record['address'] . ' ' . $record['address2'];
                            unset($record['address2']);
                        }

                        if (!isset($record['type']) || empty($record['type']))
                        {
                            $record['type'] = 'customer';
                        }

                        if (!isset($record['company_profile_id']) || empty($record['company_profile_id']) || $record['company_profile_id'] == null)
                        {
                            $record['company_profile_id'] = $cmpProfileId;
                        }

                        $record['lead_source_tag_id'] = (isset($record['lead_source_tag_id']) && $record['lead_source_tag_id'] != '') ? $record['lead_source_tag_id'] : null;


                        if ($this->validateRecord($record, $row))
                        {
                            $record['invoices_paid_by'] = 0;
                            $record['email_default']    = 0;
                            $record['user_id']          = $userId;

                            $client = Client::create($record);
                            if ($customFields)
                            {
                                foreach ($customFields as $field => $key)
                                {
                                    $customRecord[$field] = $data[$key];
                                }

                                $client->custom->update($customRecord);

                            }
                            $response['total_records'] = ($response['total_records'] + 1);
                        }
                        else
                        {
                            $response['message'] = $this->messages;
                            $this->errors        = true;
                            return $response;

                        }
                    }
                    $row++;
                }
            });

            fclose($handle);

            if ($this->errors != true && $response['total_records'] == 0)
            {
                $response['success'] = false;
                $this->errors        = true;
                $response['message'] = trans('fi.no_import_records');
                $this->messages->add('error', $response['message']);
            }

            if ($this->errors != true)
            {
                $response['success'] = true;
                $this->errors        = false;
            }
        }
        catch (\Throwable $e)
        {
            Log::error($e->getMessage());
            $response['message'] = $e->getMessage();
            $this->messages->add('error', $e->getMessage());
        }
        catch (\Exception $e)
        {
            Log::error($e->getMessage());
            $response['message'] = $e->getMessage();
            $this->messages->add('error', $e->getMessage());
        }
        return $response;
    }
}