<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Expenses\Models;

use FI\Modules\CustomFields\Support\CustomFieldsParser;
use FI\Support\CurrencyFormatter;
use FI\Support\DateFormatter;
use FI\Support\NumberFormatter;
use FI\Traits\Sortable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Carbon;

class Expense extends Model
{
    use Sortable;

    protected $table = 'expenses';

    protected $fillable = ['user_id', 'client_id', 'company_profile_id', 'expense_date', 'category_name', 'amount', 'tax', 'vendor_name', 'description', 'category_id', 'vendor_id', 'tax_paid', 'type'];
    protected $sortable = ['id', 'clients.name', 'expense_date', 'expense_categories.name', 'description', 'amount', 'tax_paid'];

    /*
    |--------------------------------------------------------------------------
    | Relationships
    |--------------------------------------------------------------------------
    */

    public function attachments()
    {
        return $this->morphMany('FI\Modules\Attachments\Models\Attachment', 'attachable');
    }

    public function category()
    {
        return $this->belongsTo('FI\Modules\Expenses\Models\ExpenseCategory');
    }

    public function client()
    {
        return $this->belongsTo('FI\Modules\Clients\Models\Client');
    }

    public function companyProfile()
    {
        return $this->belongsTo('FI\Modules\CompanyProfiles\Models\CompanyProfile');
    }

    public function custom()
    {
        return $this->hasOne('FI\Modules\CustomFields\Models\ExpenseCustom');
    }

    public function invoice()
    {
        return $this->belongsTo('FI\Modules\Invoices\Models\Invoice');
    }

    public function vendor()
    {
        return $this->belongsTo('FI\Modules\Expenses\Models\ExpenseVendor');
    }

    public function taxRate()
    {
        return $this->belongsTo('FI\Modules\TaxRates\Models\TaxRate', 'tax', 'id');
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors
    |--------------------------------------------------------------------------
    */

    public function getAttachmentPermissionOptionsAttribute()
    {
        return [
            '0' => trans('fi.not_visible'),
            '1' => trans('fi.visible'),
        ];
    }

    public static function getExpenseTypeLists()
    {
        return [
            'standard_expense'            => trans('fi.standard_expense'),
            'expense_with_deductible_tax' => trans('fi.expense_with_deductible_tax'),
            'remittance_to_tax_authority' => trans('fi.remittance_to_tax_authority'),
        ];
    }

    public function getFormattedAmountAttribute()
    {
        return CurrencyFormatter::format($this->amount);
    }

    public function getFormattedTaxAttribute()
    {
        return CurrencyFormatter::format($this->tax);
    }

    public function getTotalAttribute()
    {
        return ($this->amount + $this->tax);
    }

    public function getFormattedTotalAttribute()
    {
        return CurrencyFormatter::format($this->amount + $this->tax);
    }

    public function getFormattedExpenseDateAttribute()
    {
        return DateFormatter::format($this->expense_date);
    }

    public function getFormattedNumericAmountAttribute()
    {
        return NumberFormatter::format($this->amount);
    }

    public function getFormattedNumericTaxPaidAttribute()
    {
        return NumberFormatter::format($this->tax_paid);
    }

    public function getFormattedTaxPaidAttribute()
    {
        return CurrencyFormatter::format($this->tax_paid);
    }

    public function getHasBeenBilledAttribute()
    {
        if ($this->invoice_id)
        {
            return true;
        }

        return false;
    }

    public function getIsBillableAttribute()
    {
        if ($this->client_id)
        {
            return true;
        }

        return false;
    }

    public function getFormattedDescriptionAttribute()
    {
        if ($this->description && strlen($this->description) > 60)
        {
            return '<span>' . mb_substr($this->description, 0, 60) . '...' . '</span>';
        }
        else
        {
            return $this->description;
        }
    }

    public function getFormattedVendorNameAttribute()
    {
        return isset($this->vendor->name) ? $this->vendor->name : '';
    }

    public function getFormattedCategoryNameAttribute()
    {
        return isset($this->category->name) ? $this->category->name : '';
    }

    /*
    |--------------------------------------------------------------------------
    | Static Methods
    |--------------------------------------------------------------------------
    */

    public static function getHeaders()
    {
        return ['id', 'vendor', 'date', 'category', 'description', 'amount'];
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeCategoryId($query, $categoryId = null)
    {
        if ($categoryId)
        {
            $query->where('expenses.category_id', $categoryId);
        }

        return $query;
    }

    public function scopeCompanyProfileId($query, $companyProfileId = null)
    {
        if ($companyProfileId)
        {
            $query->where('expenses.company_profile_id', $companyProfileId);
        }

        return $query;
    }

    public function scopeDefaultQuery($query)
    {
        return $query->select('expenses.*', 'expense_categories.name AS category_name',
            'expense_vendors.name AS vendor_name', 'clients.name AS client_name')
            ->leftJoin('expense_categories', 'expense_categories.id', '=', 'expenses.category_id')
            ->leftJoin('expense_vendors', 'expense_vendors.id', '=', 'expenses.vendor_id')
            ->leftJoin('clients', 'clients.id', '=', 'expenses.client_id');
    }

    public function scopeKeywords($query, $keywords = null)
    {
        if ($keywords)
        {
            $keywords                 = strtolower($keywords);
            $currencyStringMatchScore = currencyStringMatch($keywords);

            switch ($currencyStringMatchScore)
            {
                case 0:
                    // Definitely not a currency value search
                    $query->where('expenses.description', 'like', '%' . $keywords . '%')
                        ->orWhere(function ($query) use ($keywords)
                        {
                            $formattedDate = DateFormatter::format($keywords);
                            if ($formattedDate !== null)
                            {
                                $date = Carbon::parse($formattedDate)->format('Y-m-d');
                                $query->where('expenses.expense_date', 'like', '%' . $date . '%');
                            }
                        })
                        ->orWhere('expense_vendors.name', 'like', '%' . $keywords . '%')
                        ->orWhere('clients.name', 'like', '%' . $keywords . '%')
                        ->orWhere('expense_categories.name', 'like', '%' . $keywords . '%');
                    break;

                default :
                    $decimalValue = convertToDecimalFormat($keywords);
                    $query->where('expenses.amount', 'like', '%' . $decimalValue . '%');
                    break;

            }
        }

        return $query;
    }

    public function scopeStatus($query, $status = null)
    {
        if ($status)
        {
            switch ($status)
            {
                case 'billed':
                    $query->whereNotNull('invoice_id');
                    break;
                case 'not_billed':
                    $query->where('client_id', '<>', 0)->whereNull('invoice_id');
                    break;
                case 'not_billable':
                    $query->where('client_id', 0);
                    break;
            }
        }

        return $query;
    }

    public function scopeVendorId($query, $vendorId = null)
    {
        if ($vendorId)
        {
            $query->where('vendor_id', $vendorId);
        }

        return $query;
    }

    public function scopeDateRange($query, $fromDate = null, $toDate = null)
    {
        if (isset($fromDate) && isset($toDate) && $fromDate != null && $toDate != null)
        {
            return $query->where('expenses.expense_date', '>=', $fromDate)
                ->where('expenses.expense_date', '<=', $toDate);
        }
    }

    public function scopeType($query, $type = null)
    {
        if ($type)
        {
            $query->where('expenses.type', $type);
        }
        return $query;
    }

    public function scopeClientId($query, $clientId)
    {
        if ($clientId)
        {
            $query->whereHas('client', function ($query) use ($clientId)
            {
                $query->where('id', $clientId);
            });
        }

        return $query;
    }

    /*
    |--------------------------------------------------------------------------
    | Other
    |--------------------------------------------------------------------------
    */

    public function customField($label, $rawHtml = true)
    {
        $customField = config('fi.customFields')->where('tbl_name', 'expenses')->where('field_label', $label)->first();

        if ($customField)
        {
            return CustomFieldsParser::getFieldValue($this->custom, $customField, $rawHtml);
        }

        return null;

    }
}