<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\DocumentRevisions\Support;


use FI\Modules\CustomFields\Support\CustomFieldsParser;
use FI\Modules\DocumentNumberSchemes\Models\DocumentNumberScheme;
use FI\Modules\DocumentRevisions\Models\DocumentRevision;
use Milon\Barcode\DNS2D;

class DocumentRevisionSupport
{

    public static function findModule($type)
    {
        switch ($type)
        {
            case 'quote':
                return 'FI\Modules\Quotes\Models\Quote';
            case 'invoice':
                return 'FI\Modules\Invoices\Models\Invoice';
            case 'proformaInvoice':
                return 'Addons\ProformaInvoice\Models\ProformaInvoice';
            case 'subscription':
                return 'FI\Modules\RecurringInvoices\Models\RecurringInvoice';
            default:
                return null;
        }
    }

    public function addData($type, $data)
    {
        $type = self::findModule($type);

        if ($type)
        {
            $details = $this->setData($type, $data);
            DocumentRevision::create($details);
        }
        else
        {
            return;
        }
    }

    public function setData($type, $data)
    {
        $documentRevision = DocumentRevision::where('revisable_type', $type)->where('revisable_id', $data['id']);

        $getRevisionNumber = self::getRevisionNumber($type, $data['id']);

        $dataNew = [
            'revision_number' => $getRevisionNumber,
            'revisable_id'    => $data['id'],
            'revisable_type'  => $type,
            'text'            => json_encode($data, JSON_UNESCAPED_UNICODE),
            'updated_by'      => auth()->id()
        ];

        $revisionCount = $documentRevision->count();

        if ($revisionCount >= 10)
        {
            $documentRevision->orderBy('id')->first()->delete();
        }

        return $dataNew;
    }

    public static function getRevisionNumber($type, $id)
    {
        $documentRevision = DocumentRevision::select('revision_number')->where('revisable_type', $type)->where('revisable_id', $id)->orderBy('revision_number', 'DESC')->first();
        return (isset($documentRevision->revision_number) && !empty($documentRevision)) ? (intVal($documentRevision->revision_number) + 1) : 1;
    }

    public function prepareArrayForInvoice($invoice)
    {
        $selectedTags = [];

        foreach ($invoice->tags as $tagDetail)
        {
            $selectedTags[] = $tagDetail->tag->name;
        }

        $rd_inv                                  = [];
        $invoiceItemCustomFields                 = CustomFieldsParser::getFields('invoice_items');
        $invoiceCustomFields                     = CustomFieldsParser::getFields('invoices');
        $rd_inv['id']                            = $invoice->id;
        $rd_inv['type']                          = $invoice->type;
        $rd_inv['client_language']               = $invoice->client->language;
        $rd_inv['footer']                        = $invoice->formatted_footer;
        $rd_inv['currency_code']                 = $invoice->currency_code;
        $rd_inv['status']                        = $invoice->status;
        $rd_inv['template']                      = $invoice->template;
        $rd_inv['exchange_rate']                 = $invoice->exchange_rate;
        $rd_inv['recurring_invoice_id']          = $invoice->recurring_invoice_id;
        $rd_inv['service_explanation']           = $invoice->service_explanation;
        $rd_inv['formatted_service_starts_on']   = $invoice->formatted_service_starts_on;
        $rd_inv['formatted_service_ends_off']    = $invoice->formatted_service_ends_off;
        $rd_inv['terms']                         = $invoice->formatted_terms;
        $rd_inv['url_key']                       = $invoice->url_key;
        $rd_inv['summary']                       = $invoice->summary;
        $rd_inv['discount']                      = $invoice->discount;
        $rd_inv['numeric_discount']              = $invoice->formatted_numeric_discount;
        $rd_inv['formatted_discount']            = $invoice->amount->formatted_discount;
        $rd_inv['discount_type']                 = $invoice->discount_type;
        $rd_inv['type_title']                    = ($invoice['type'] == 'credit_memo') ? trans('fi.credit_memo') : trans('fi.invoice');
        $rd_inv['formatted_created_at']          = $invoice->formatted_created_at;
        $rd_inv['number']                        = $invoice->number;
        $rd_inv['due_at']                        = $invoice->formatted_due_at;
        $rd_inv['formatted_invoice_date']        = $invoice->formatted_invoice_date;
        $rd_inv['selected_tags']                 = $selectedTags;
        $rd_inv['bill_to']                       = $invoice->client->title != '' ? $invoice->client->title . ' ' . $invoice->client->name : $invoice->client->name;
        $rd_inv['client_name']                   = $invoice->client->name;
        $rd_inv['client_phone']                  = $invoice->client->phone;
        $rd_inv['client_email']                  = $invoice->client->email;
        $rd_inv['date_emailed']                  = $invoice->formatted_date_emailed;
        $rd_inv['date_mailed']                   = $invoice->formatted_date_mailed;
        $rd_inv['allow_line_item_discounts']     = ($invoice->type != 'credit_memo' && config('fi.allowLineItemDiscounts') == 1) ? true : false;
        $rd_inv['online_payment_processing_fee'] = ($invoice->online_payment_processing_fee == 'yes') ? trans('fi.yes') : trans('fi.no');
        $rd_inv['has_line_item_discount']        = $invoice->hasLineItemDiscount();
        $rd_inv['summarized_taxes']              = $invoice->summarized_taxes;
        $rd_inv['client_formatted_address']      = $invoice->client->formatted_address;
        if (config('fi.clientColumnSettingsVatTaxId') == 1)
        {
            if ($invoice->client->vat_tax_id)
            {
                $rd_inv['vat_tax_id'] = $invoice->client->vat_tax_id;
            }
        }
        if (config('fi.qrCodeOnInvoiceQuote') == 1)
        {
            $rd_inv['qr_code'] = ((new DNS2D())->getBarcodePNG(route('clientCenter.public.invoice.show', [$invoice->url_key, $invoice->token]), "QRCODE"));
        }
        $rd_inv['company_logo']    = $invoice->companyProfile->logo();
        $rd_inv['company']         = $invoice->companyProfile->company;
        $rd_inv['company_address'] = $invoice->companyProfile->formatted_address;
        $rd_inv['company_phone']   = $invoice->companyProfile->phone;
        if (config('fi.clientColumnSettingsVatTaxId') == 1)
        {
            $rd_inv['company_vat_detail'] = nl2br($invoice->companyProfile->vat_detail);
        }
        if (isset($invoice->user->fromEmail))
        {
            $rd_inv['user_from_email'] = $invoice->user->fromEmail;
        }
        $rd_inv_custom = [];
        if (count($invoiceCustomFields))
        {
            foreach ($invoiceCustomFields as $invoiceCustomField)
            {
                $rd_inv_custom[] = [ucfirst($invoiceCustomField->field_label) => ($invoice->custom->{$invoiceCustomField->column_name}) ?? "-"];
            }
        }
        $rd_inv['custom']                   = $rd_inv_custom;
        $rd_inv['total']                    = $invoice->amount->formatted_total;
        $rd_inv['paid_label']               = ($invoice->type == 'credit_memo') ? trans('fi.applied') : trans('fi.paid');
        $rd_inv['paid']                     = $invoice->amount->formatted_paid;
        $rd_inv['balance']                  = $invoice->amount->formatted_balance;
        $rd_inv['is_payment_term']          = isset($invoice->paymentTerm->id) ? true : false;
        $rd_inv['payment_term_description'] = $rd_inv['is_payment_term'] ? $invoice->formatted_actual_terms_text : null;
        $rd_inv['payment_term_name']        = $rd_inv['is_payment_term'] ? $invoice->paymentTerm->name : '';
        $rd_inv['payment_term_is_default']  = $invoice->paymentTermsIsDefault;

        foreach ($invoice->items as $key => $item)
        {
            $customItm = [];
            if (count($invoiceItemCustomFields))
            {
                foreach ($invoiceItemCustomFields as $invoiceItemCustomField)
                {
                    $customItm[] = [ucfirst($invoiceItemCustomField->field_label) => ($item->custom->{$invoiceItemCustomField->column_name}) ?? "-"];
                }
            }

            $rd_inv['items'][$item->id] = [
                'id'                 => $item->id,
                'name'               => $item->name,
                'description'        => $item->formatted_description,
                'quantity'           => $item->formatted_quantity,
                'price'              => $item->formatted_price,
                'numeric_price'      => $item->formatted_numeric_price,
                'discount_type'      => $item->discount_type,
                'numeric_discount'   => $item->formatted_numeric_discount,
                'formatted_discount' => $item->formatted_discount,
                'tax_rate_id'        => $item->tax_rate_id,
                'tax_rate_2_id'      => $item->tax_rate_2_id,
                'item_subtotal'      => $item->amount->formatted_subtotal,
                'custom'             => $customItm,
            ];
        }
        $rd_inv['subtotal'] = $invoice->amount->formatted_subtotal;

        if ($invoice->amount->payment_terms_discount > 0)
        {
            $rd_inv['term_discount'] = $invoice->amount->formatted_payment_terms_discount;
        }
        $rd_inv['tax'] = $invoice->amount->formatted_tax;

        if ($invoice->total_convenience_charges > 0)
        {
            $rd_inv['convenience_charges'] = $invoice->formatted_total_convenience_charges;
        }

        return ['type' => 'invoice', 'data' => $rd_inv];
    }

    public function prepareArrayForQuote($quote)
    {
        $selectedTags = [];

        foreach ($quote->tags as $tagDetail)
        {
            $selectedTags[] = $tagDetail->tag->name;
        }

        $rd_quote              = [];
        $quoteItemCustomFields = CustomFieldsParser::getFields('quote_items');
        $quoteCustomFields     = CustomFieldsParser::getFields('quotes');

        $rd_quote['id']                            = $quote->id;
        $rd_quote['number']                        = $quote->number;
        $rd_quote['summary']                       = $quote->summary;
        $rd_quote['viewed']                        = $quote->viewed;
        $rd_quote['is_invoice']                    = isset($quote->invoice->id);
        $rd_quote['selected_tags']                 = $selectedTags;
        $rd_quote['client_language']               = $quote->client->language;
        $rd_quote['footer']                        = $quote->formatted_footer;
        $rd_quote['terms']                         = $quote->formatted_terms;
        $rd_quote['currency_code']                 = $quote->currency_code;
        $rd_quote['subtotal']                      = $quote->amount->formatted_subtotal;
        $rd_quote['template']                      = $quote->template;
        $rd_quote['invoice_id']                    = $quote->invoice_id;
        $rd_quote['invoice_number']                = isset($quote->invoice->id) ? $quote->invoice->number : null;
        $rd_quote['status']                        = $quote->status;
        $rd_quote['exchange_rate']                 = $quote->exchange_rate;
        $rd_quote['url_key']                       = $quote->url_key;
        $rd_quote['discount']                      = $quote->discount;
        $rd_quote['tax']                           = $quote->amount->formatted_tax;
        $rd_quote['numeric_discount']              = $quote->formatted_numeric_discount;
        $rd_quote['discount_type']                 = $quote->discount_type;
        $rd_quote['formatted_discount']            = $quote->amount->formatted_discount;
        $rd_quote['formatted_created_at']          = $quote->formatted_created_at;
        $rd_quote['expires_at']                    = $quote->formatted_expires_at;
        $rd_quote['formatted_quote_date']          = $quote->formatted_quote_date;
        $rd_quote['bill_to']                       = $quote->client->title != '' ? $quote->client->title . ' ' . $quote->client->name : $quote->client->name;
        $rd_quote['client_name']                   = $quote->client->name;
        $rd_quote['client_phone']                  = $quote->client->phone;
        $rd_quote['client_email']                  = $quote->client->email;
        $rd_quote['client_formatted_address']      = $quote->client->formatted_address;
        $rd_quote['allow_line_item_discounts']     = (config('fi.allowLineItemDiscounts') == 1) ? true : false;
        $rd_quote['has_line_item_discount']        = $quote->hasLineItemDiscount();
        $rd_quote['summarized_taxes']              = $quote->summarized_taxes;
        $rd_quote['setting_is_payable_quote']      = config('fi.isPayableQuote', 0);
        $rd_quote['is_payable_quote']              = $quote->is_payable_quote;
        $rd_quote['online_payment_processing_fee'] = $quote->online_payment_processing_fee;
        $rd_quote['enableOppFees']                 = config('fi.enableOppFees');
        if (config('fi.clientColumnSettingsVatTaxId') == 1)
        {
            if ($quote->client->vat_tax_id)
            {
                $rd_quote['vat_tax_id'] = $quote->client->vat_tax_id;
            }
        }
        if (config('fi.qrCodeOnInvoiceQuote') == 1)
        {
            $rd_quote['qr_code'] = ((new DNS2D())->getBarcodePNG(route('clientCenter.public.quote.show', [$quote->url_key, $quote->token]), "QRCODE"));
        }
        $rd_quote['company_logo']    = $quote->companyProfile->logo();
        $rd_quote['company']         = $quote->companyProfile->company;
        $rd_quote['company_address'] = $quote->companyProfile->formatted_address;
        $rd_quote['company_phone']   = $quote->companyProfile->phone;
        if (config('fi.clientColumnSettingsVatTaxId') == 1)
        {
            $rd_quote['company_vat_detail'] = nl2br($quote->companyProfile->vat_detail);
        }
        if (isset($quote->user->fromEmail))
        {
            $rd_quote['user_from_email'] = $quote->user->fromEmail;
        }
        $rd_quote_custom = [];
        if (count($quoteCustomFields))
        {
            foreach ($quoteCustomFields as $quoteCustomField)
            {
                $rd_quote_custom[] = [ucfirst($quoteCustomField->field_label) => ($quote->custom->{$quoteCustomField->column_name}) ?? "-"];
            }
        }
        $rd_quote['custom']  = $rd_quote_custom;
        $rd_quote['total']   = $quote->amount->formatted_total;
        $rd_quote['paid']    = $quote->amount->formatted_paid;
        $rd_quote['balance'] = $quote->amount->formatted_balance;

        foreach ($quote->items as $key => $item)
        {
            $customItm = [];
            if (count($quoteItemCustomFields))
            {
                foreach ($quoteItemCustomFields as $quoteItemCustomField)
                {
                    $customItm[] = [ucfirst($quoteItemCustomField->field_label) => ($item->custom->{$quoteItemCustomField->column_name}) ?? "-"];
                }
            }

            $rd_quote['items'][$item->id] = [
                'id'                 => $item->id,
                'name'               => $item->name,
                'description'        => $item->formatted_description,
                'quantity'           => $item->formatted_quantity,
                'price'              => $item->formatted_price,
                'numeric_price'      => $item->formatted_numeric_price,
                'discount_type'      => $item->discount_type,
                'numeric_discount'   => $item->formatted_numeric_discount,
                'formatted_discount' => $item->formatted_discount,
                'tax_rate_id'        => $item->tax_rate_id,
                'tax_rate_2_id'      => $item->tax_rate_2_id,
                'item_subtotal'      => $item->amount->formatted_subtotal,
                'custom'             => $customItm,
            ];
        }

        return ['type' => 'quote', 'data' => $rd_quote];
    }

    public function prepareArrayForRecurringInvoice($recurringInvoice)
    {
        $selectedTags = [];

        foreach ($recurringInvoice->tags as $tagDetail)
        {
            $selectedTags[] = $tagDetail->tag->name;
        }
        $rd_ri                        = [];
        $recurringInvoiceCustomFields = CustomFieldsParser::getFields('recurring_invoices');
        $rd_ri_custom                 = [];
        if (count($recurringInvoiceCustomFields))
        {
            foreach ($recurringInvoiceCustomFields as $recurringInvoiceCustomField)
            {
                $rd_ri_custom[] = [ucfirst($recurringInvoiceCustomField->field_label) => ($recurringInvoice->custom->{$recurringInvoiceCustomField->column_name}) ?? "-"];
            }
        }
        $rd_ri['custom'] = $rd_ri_custom;

        $recurringInvoiceItemCustomFields = CustomFieldsParser::getFields('recurring_invoice_items');
        foreach ($recurringInvoice->items as $key => $item)
        {
            $customItm = [];
            if (count($recurringInvoiceItemCustomFields))
            {
                foreach ($recurringInvoiceItemCustomFields as $recurringInvoiceItemCustomField)
                {
                    $customItm[] = [ucfirst($recurringInvoiceItemCustomField->field_label) => ($item->custom->{$recurringInvoiceItemCustomField->column_name}) ?? "-"];
                }
            }

            $rd_ri['items'][$item->id] = [
                'id'            => $item->id,
                'name'          => $item->name,
                'description'   => $item->formatted_description,
                'quantity'      => $item->formatted_quantity,
                'numeric_price' => $item->formatted_numeric_price,
                'tax_rate_id'   => $item->tax_rate_id,
                'tax_rate_2_id' => $item->tax_rate_2_id,
                'item_subtotal' => $item->amount->formatted_subtotal,
                'price'         => $item->formatted_price,
                'custom'        => $customItm,
            ];
        }
        $rd_ri['id']                       = $recurringInvoice->id;
        $rd_ri['number']                   = $recurringInvoice->id;
        $rd_ri['recurring_frequency']      = $recurringInvoice->recurring_frequency;
        $rd_ri['recurring_period']         = $recurringInvoice->recurring_period;
        $rd_ri['formatted_next_date']      = $recurringInvoice->formatted_next_date;
        $rd_ri['formatted_stop_date']      = $recurringInvoice->formatted_stop_date;
        $rd_ri['service_explanation']      = $recurringInvoice->service_explanation;
        $rd_ri['service_starts_on']        = $recurringInvoice->service_starts_on;
        $rd_ri['service_ends_of_the']      = $recurringInvoice->service_ends_of_the;
        $rd_ri['summary']                  = $recurringInvoice->summary;
        $rd_ri['selected_tags']            = $selectedTags;
        $rd_ri['company_logo']             = $recurringInvoice->companyProfile->logo();
        $rd_ri['company']                  = $recurringInvoice->companyProfile->company;
        $rd_ri['company_address']          = $recurringInvoice->companyProfile->formatted_address;
        $rd_ri['company_phone']            = $recurringInvoice->companyProfile->phone;
        $rd_ri['client_name']              = $recurringInvoice->client->name;
        $rd_ri['client_phone']             = $recurringInvoice->client->phone;
        $rd_ri['client_email']             = $recurringInvoice->client->email;
        $rd_ri['client_formatted_address'] = $recurringInvoice->client->formatted_address;
        $rd_ri['terms']                    = $recurringInvoice->terms;
        $rd_ri['footer']                   = $recurringInvoice->footer;

        if (isset($recurringInvoice->user->fromEmail))
        {
            $rd_ri['user_from_email'] = $recurringInvoice->user->fromEmail;
        }

        $rd_ri['subtotal']               = $recurringInvoice->amount->formatted_subtotal;
        $rd_ri['formatted_discount']     = $recurringInvoice->amount->formatted_discount;
        $rd_ri['tax']                    = $recurringInvoice->amount->formatted_tax;
        $rd_ri['total']                  = $recurringInvoice->amount->formatted_total;
        $rd_ri['balance']                = $recurringInvoice->amount->formatted_balance;
        $rd_ri['url_key']                = $recurringInvoice->url_key;
        $rd_ri['discount']               = $recurringInvoice->discount;
        $rd_ri['numeric_discount']       = $recurringInvoice->formatted_numeric_discount;
        $rd_ri['discount_type']          = $recurringInvoice->discount_type;
        $rd_ri['currency_code']          = $recurringInvoice->currency_code;
        $rd_ri['template']               = $recurringInvoice->template;
        $rd_ri['exchange_rate']          = $recurringInvoice->exchange_rate;
        $documentNumberScheme            = DocumentNumberScheme::getList();
        $rd_ri['document_number_scheme'] = $documentNumberScheme[$recurringInvoice->document_number_scheme_id];

        return ['type' => 'subscription', 'data' => $rd_ri];
    }

    public function prepareArrayForProformaInvoice($proformaInvoice)
    {
        $selectedTags = [];

        foreach ($proformaInvoice->tags as $tagDetail)
        {
            $selectedTags[] = $tagDetail->tag->name;
        }

        $rd_pi                           = [];
        $proformaInvoiceItemCustomFields = CustomFieldsParser::getFields('proforma_invoice_items');
        $proformaInvoiceCustomFields     = CustomFieldsParser::getFields('proforma_invoices');

        $rd_pi['id']                              = $proformaInvoice->id;
        $rd_pi['number']                          = $proformaInvoice->number;
        $rd_pi['summary']                         = $proformaInvoice->summary;
        $rd_pi['viewed']                          = $proformaInvoice->viewed;
        $rd_pi['is_invoice']                      = isset($proformaInvoice->invoice->id);
        $rd_pi['selected_tags']                   = $selectedTags;
        $rd_pi['client_language']                 = $proformaInvoice->client->language;
        $rd_pi['footer']                          = $proformaInvoice->formatted_footer;
        $rd_pi['terms']                           = $proformaInvoice->formatted_terms;
        $rd_pi['currency_code']                   = $proformaInvoice->currency_code;
        $rd_pi['subtotal']                        = $proformaInvoice->amount->formatted_subtotal;
        $rd_pi['template']                        = $proformaInvoice->template;
        $rd_pi['invoice_id']                      = $proformaInvoice->invoice_id;
        $rd_pi['invoice_number']                  = isset($proformaInvoice->invoice->id) ? $proformaInvoice->invoice->number : null;
        $rd_pi['status']                          = $proformaInvoice->status;
        $rd_pi['exchange_rate']                   = $proformaInvoice->exchange_rate;
        $rd_pi['url_key']                         = $proformaInvoice->url_key;
        $rd_pi['discount']                        = $proformaInvoice->discount;
        $rd_pi['tax']                             = $proformaInvoice->amount->formatted_tax;
        $rd_pi['numeric_discount']                = $proformaInvoice->formatted_numeric_discount;
        $rd_pi['discount_type']                   = $proformaInvoice->discount_type;
        $rd_pi['formatted_discount']              = $proformaInvoice->amount->formatted_discount;
        $rd_pi['formatted_created_at']            = $proformaInvoice->formatted_created_at;
        $rd_pi['expires_at']                      = $proformaInvoice->formatted_expires_at;
        $rd_pi['formatted_proforma_invoice_date'] = $proformaInvoice->formatted_proforma_invoice_date;
        $rd_pi['bill_to']                         = $proformaInvoice->client->title != '' ? $proformaInvoice->client->title . ' ' . $proformaInvoice->client->name : $proformaInvoice->client->name;
        $rd_pi['client_name']                     = $proformaInvoice->client->name;
        $rd_pi['client_phone']                    = $proformaInvoice->client->phone;
        $rd_pi['client_email']                    = $proformaInvoice->client->email;
        $rd_pi['client_formatted_address']        = $proformaInvoice->client->formatted_address;
        $rd_pi['allow_line_item_discounts']       = (config('fi.allowLineItemDiscounts') == 1) ? true : false;
        $rd_pi['has_line_item_discount']          = $proformaInvoice->hasLineItemDiscount();
        $rd_pi['summarized_taxes']                = $proformaInvoice->summarized_taxes;
        if (config('fi.clientColumnSettingsVatTaxId') == 1)
        {
            if ($proformaInvoice->client->vat_tax_id)
            {
                $rd_pi['vat_tax_id'] = $proformaInvoice->client->vat_tax_id;
            }
        }
        if (config('fi.qrCodeOnInvoiceQuote') == 1)
        {
            $rd_pi['qr_code'] = ((new DNS2D())->getBarcodePNG(route('clientCenter.public.quote.show', [$proformaInvoice->url_key, $proformaInvoice->token]),"QRCODE"));
        }
        $rd_pi['company_logo']    = $proformaInvoice->companyProfile->logo();
        $rd_pi['company']         = $proformaInvoice->companyProfile->company;
        $rd_pi['company_address'] = $proformaInvoice->companyProfile->formatted_address;
        $rd_pi['company_phone']   = $proformaInvoice->companyProfile->phone;
        if (config('fi.clientColumnSettingsVatTaxId') == 1)
        {
            $rd_pi['company_vat_detail'] = nl2br($proformaInvoice->companyProfile->vat_detail);
        }
        if (isset($proformaInvoice->user->fromEmail))
        {
            $rd_pi['user_from_email'] = $proformaInvoice->user->fromEmail;
        }
        $rd_pi_custom = [];
        if (count($proformaInvoiceCustomFields))
        {
            foreach ($proformaInvoiceCustomFields as $proformaInvoiceCustomField)
            {
                $rd_pi_custom[] = [ucfirst($proformaInvoiceCustomField->field_label) => ($proformaInvoice->custom->{$proformaInvoiceCustomField->column_name}) ?? "-"];
            }
        }
        $rd_pi['custom']  = $rd_pi_custom;
        $rd_pi['total']   = $proformaInvoice->amount->formatted_total;
        $rd_pi['paid']    = $proformaInvoice->amount->formatted_paid;
        $rd_pi['balance'] = $proformaInvoice->amount->formatted_balance;

        foreach ($proformaInvoice->items as $key => $item)
        {
            $customItm = [];
            if (count($proformaInvoiceItemCustomFields))
            {
                foreach ($proformaInvoiceItemCustomFields as $proformaInvoiceItemCustomField)
                {
                    $customItm[] = [ucfirst($proformaInvoiceItemCustomField->field_label) => ($item->custom->{$proformaInvoiceItemCustomField->column_name}) ?? "-"];
                }
            }

            $rd_pi['items'][$item->id] = [
                'id'                 => $item->id,
                'name'               => $item->name,
                'description'        => $item->formatted_description,
                'quantity'           => $item->formatted_quantity,
                'price'              => $item->formatted_price,
                'numeric_price'      => $item->formatted_numeric_price,
                'discount_type'      => $item->discount_type,
                'numeric_discount'   => $item->formatted_numeric_discount,
                'formatted_discount' => $item->formatted_discount,
                'tax_rate_id'        => $item->tax_rate_id,
                'tax_rate_2_id'      => $item->tax_rate_2_id,
                'item_subtotal'      => $item->amount->formatted_subtotal,
                'custom'             => $customItm,
            ];
        }

        return ['type' => 'proformaInvoice', 'data' => $rd_pi];
    }

}