<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\DocumentNumberSchemes\Controllers;

use FI\Http\Controllers\Controller;
use FI\Modules\DocumentNumberSchemes\DocumentNumberSchemeOptions;
use FI\Modules\DocumentNumberSchemes\Models\DocumentNumberScheme;
use FI\Modules\DocumentNumberSchemes\Requests\DocumentNumberSchemeRequest;
use FI\Traits\ReturnUrl;

class DocumentNumberSchemeController extends Controller
{
    use ReturnUrl;

    private $documentNumberSchemeOptions;

    public function __construct(DocumentNumberSchemeOptions $documentNumberSchemeOptions)
    {
        $this->documentNumberSchemeOptions = $documentNumberSchemeOptions;
    }

    public function index()
    {
        $this->setReturnUrl();

        $documentNumberSchemes = DocumentNumberScheme::sortable(['name' => 'asc'])->paginate(config('fi.resultsPerPage'));


        return view('document_number_schemes.index')
            ->with('documentNumberSchemes', $documentNumberSchemes)
            ->with('resetNumberOptions', $this->documentNumberSchemeOptions->resetNumberOptions());
    }

    public function create()
    {
        return view('document_number_schemes.form')
            ->with('editMode', false)
            ->with('types', !config('proforma_invoice_enabled') ? ['' => trans('fi.select_type'), 'invoice' => trans('fi.invoice_default'), 'quote' => trans('fi.quote_default'), 'credit_memo' => trans('fi.credit_memo_default')] : ['' => trans('fi.select_type'), 'invoice' => trans('fi.invoice_default'), 'quote' => trans('fi.quote_default'), 'credit_memo' => trans('fi.credit_memo_default'), 'proforma_invoice' => trans('fi.proforma_invoice_default'), 'support_ticket' => trans('fi.support_ticket')])
            ->with('resetNumberOptions', $this->documentNumberSchemeOptions->resetNumberOptions());
    }

    public function store(DocumentNumberSchemeRequest $request)
    {
        DocumentNumberScheme::create($request->all());

        return redirect($this->getReturnUrl())
            ->with('alertSuccess', trans('fi.record_successfully_created'));
    }

    public function edit($id)
    {
        $documentNumberScheme = DocumentNumberScheme::find($id);

        return view('document_number_schemes.form')
            ->with('editMode', true)
            ->with('documentNumberScheme', $documentNumberScheme)
            ->with('types', DocumentNumberScheme::getTypes())
            ->with('resetNumberOptions', $this->documentNumberSchemeOptions->resetNumberOptions());
    }

    public function update(DocumentNumberSchemeRequest $request, $id)
    {
        $documentNumberScheme = DocumentNumberScheme::find($id);

        $documentNumberScheme->fill($request->all());

        $documentNumberScheme->save();

        return redirect($this->getReturnUrl())
            ->with('alertSuccess', trans('fi.record_successfully_updated'));
    }

    public function delete($id)
    {
        try
        {
            DocumentNumberScheme::destroy($id);

            return response()->json(['success' => true, 'message' => trans('fi.record_successfully_deleted')], 200);
        }
        catch (\Exception $e)
        {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 400);
        }

    }

    public function deleteModal()
    {
        try
        {
            return view('layouts._delete_modal')->with('url', request('action'))->with('modalName', request('modalName'))->with('isReload', request('isReload'));
        }
        catch (\Exception $e)
        {
            return response()->json(['success' => false, 'message' => ($e->getCode() == 0) ? trans('fi.modal_not_found') : $e->getMessage()], 400);
        }
    }
}