<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\Contacts\Models;

use FI\Modules\Clients\Models\Client;
use FI\Modules\Countries\Models\Country;
use Illuminate\Database\Eloquent\Model;

class Contact extends Model
{
    protected $guarded = ['id'];

    /*
    |--------------------------------------------------------------------------
    | Static Methods
    |--------------------------------------------------------------------------
    */
    public static function getContactTitle()
    {
        return [
            'mr'   => trans('fi.mr'),
            'miss' => trans('fi.miss'),
            'ms'   => trans('fi.ms'),
            'mrs'  => trans('fi.mrs'),
            'dr'   => trans('fi.dr'),
            'prof' => trans('fi.prof'),
        ];
    }

    public static function getDefaultStatusListForToBccAndCc()
    {
        return ['0' => trans('fi.no'), '1' => trans('fi.yes_all'), '2' => trans('fi.invoices_only'), '3' => trans('fi.payment_receipts_only')];
    }

    /*
    |--------------------------------------------------------------------------
    | Relationships
    |--------------------------------------------------------------------------
    */

    public function client()
    {
        return $this->belongsTo('FI\Modules\Clients\Models\Client');
    }

    public function tags()
    {
        return $this->hasMany('FI\Modules\Contacts\Models\ContactTag');
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors
    |--------------------------------------------------------------------------
    */

    public function getFormattedContactAttribute()
    {
        return $this->name . ' <' . $this->email . '>';
    }

    public function getFormattedDefaultBccAttribute()
    {
        return ($this->default_bcc) ? trans('fi.yes') : trans('fi.no');
    }

    public function getFormattedDefaultCcAttribute()
    {
        return ($this->default_cc) ? trans('fi.yes') : trans('fi.no');
    }

    public function getFormattedDefaultToAttribute()
    {
        return ($this->default_to) ? trans('fi.yes') : trans('fi.no');
    }

    public function getFormattedNotesAttribute()
    {
        if ($this->notes && strlen($this->notes) > 40)
        {
            return '<span data-toggle="tooltip" data-placement="auto" data-original-title="' . $this->notes . '">' . mb_substr($this->notes, 0, 40) . '...' . '</span>';
        }
        elseif ($this->notes && strlen($this->notes) < 40)
        {
            return $this->notes;
        }
        else
        {
            return null;
        }
    }

    public function getFormattedStatusAttribute()
    {
        return ($this->status) ? trans('fi.yes') : trans('fi.no');
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeKeywords($query, $keyword)
    {

        if ($keyword)
        {
            $keyword = strtolower($keyword);
            $query->where('contacts.title', 'like', '%' . $keyword . '%')
                ->orWhere('contacts.email', 'like', '%' . $keyword . '%')
                ->orWhere('contacts.primary_phone', 'like', '%' . $keyword . '%')
                ->orWhere('contacts.alternate_phone', 'like', '%' . $keyword . '%')
                ->orWhere('clients.name', 'like', '%' . $keyword . '%');
        }

    }

    /*
    |--------------------------------------------------------------------------
    | Other
    |--------------------------------------------------------------------------
    */

    public static function getClientContactList($client_id)
    {
        return Contact::whereClientId($client_id)->whereStatus(1)->pluck('name', 'id')->toArray();
    }

    public static function getContactDefaultCountry($client_id)
    {
        $client = Client::find($client_id);

        $defaultCountry = 'us';
        if ($client->country)
        {
            $country        = Country::where('name', $client->country)->first();
            $defaultCountry = $country->code;
        }
        elseif ($client->companyProfile->country)
        {
            $country        = Country::where('name', $client->companyProfile->country)->first();
            $defaultCountry = $country->code;
        }

        return strtolower($defaultCountry);
    }

}