<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace FI\Modules\CompanyProfiles\Controllers;

use Exception;
use FI\Http\Controllers\Controller;
use FI\Modules\CompanyProfiles\Models\CompanyProfile;
use FI\Modules\CompanyProfiles\Requests\CompanyProfileStoreRequest;
use FI\Modules\CompanyProfiles\Requests\CompanyProfileUpdateRequest;
use FI\Modules\Countries\Models\Country;
use FI\Modules\CustomFields\Models\CompanyProfileCustom;
use FI\Modules\CustomFields\Support\CustomFieldsParser;
use FI\Modules\CustomFields\Support\CustomFieldsTransformer;
use FI\Modules\Invoices\Support\InvoiceTemplates;
use FI\Modules\PaymentTerms\Models\PaymentTerm;
use FI\Modules\Quotes\Support\QuoteTemplates;
use FI\Modules\Settings\Models\Setting;
use FI\Traits\ReturnUrl;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\Drivers\Gd\Driver;
use Intervention\Image\ImageManager;

class CompanyProfileController extends Controller
{
    use ReturnUrl;

    public function index()
    {
        $this->setReturnUrl();

        return view('company_profiles.index')
            ->with('companyProfiles', CompanyProfile::orderBy('company')->paginate(config('fi.resultsPerPage')));
    }

    public function create()
    {
        $emailList                    = Setting::emailList();
        $proformaInvoiceTemplateLists = (class_exists(\Addons\ProformaInvoice\Support\ProformaInvoiceTemplates::class));

        return view('company_profiles.form')
            ->with('editMode', false)
            ->with('proformaInvoiceTemplates', $proformaInvoiceTemplateLists == true ? \Addons\ProformaInvoice\Support\ProformaInvoiceTemplates::lists() : null)
            ->with('invoiceTemplates', InvoiceTemplates::lists())
            ->with('quoteTemplates', QuoteTemplates::lists())
            ->with('proformaInvoiceTemplates', $proformaInvoiceTemplateLists == true ? \Addons\ProformaInvoice\Support\ProformaInvoiceTemplates::lists() : null)
            ->with('proformaInvoiceEmailType', $emailList['proformaInvoice']['MailTemplate'])
            ->with('proformaInvoiceApprovedEmailType', $emailList['proformaInvoiceApproved']['MailTemplate'])
            ->with('proformaInvoiceRejectedEmailType', $emailList['proformaInvoiceRejected']['MailTemplate'])
            ->with('quoteEmailType', $emailList['quote']['MailTemplate'])
            ->with('quoteApprovedEmailType', $emailList['quoteApproved']['MailTemplate'])
            ->with('quoteRejectedEmailType', $emailList['quoteRejected']['MailTemplate'])
            ->with('invoiceEmailType', $emailList['invoice']['MailTemplate'])
            ->with('creditMemosEmailType', $emailList['creditMemos']['MailTemplate'])
            ->with('overdueInvoiceEmailType', $emailList['overdueInvoice']['MailTemplate'])
            ->with('upcomingPaymentEmailType', $emailList['upcomingPayment']['MailTemplate'])
            ->with('paymentReceiptEmailType', $emailList['paymentReceipt']['MailTemplate'])
            ->with('countries', Country::getAll())
            ->with('paymentTerms', PaymentTerm::getDropDownList())
            ->with('customFields', CustomFieldsParser::getFields('company_profiles'));
    }

    public function store(CompanyProfileStoreRequest $request)
    {
        $input   = $request->except('custom', 'quote_email_body', 'quote_approved_email_body', 'quote_rejected_email_body', 'invoice_email_body', 'credit_memos_email_body', 'overdue_invoice_email_body', 'payment_receipt_email_body', 'upcoming_payment_email_body');
        $manager = new ImageManager(new Driver());

        if ($request->hasFile('logo'))
        {
            $logoFileName = $request->file('logo')->getClientOriginalName();
            $request->file('logo')->move(company_profile_logo_path(), $logoFileName);
            if (file_exists(company_profile_logo_path() . DIRECTORY_SEPARATOR . $logoFileName))
            {
                ini_set('memory_limit', '256M');
                $manager->read(company_profile_logo_path() . DIRECTORY_SEPARATOR . $logoFileName)->resizeDown(1000, 1000)->save(company_profile_logo_path() . DIRECTORY_SEPARATOR . $logoFileName);
            }
            $input['logo'] = $logoFileName;
        }

        $companyProfile = CompanyProfile::create($input);

        if (isset($companyProfile->proforma_invoice_email_body) && config('proforma_invoice_enabled') == true)
        {
            $companyProfile->proforma_invoice_email_body          = $request->proforma_invoice_email_body . '.blade.php';
            $companyProfile->proforma_invoice_approved_email_body = $request->proforma_invoice_approved_email_body . '.blade.php';
            $companyProfile->proforma_invoice_rejected_email_body = $request->proforma_invoice_rejected_email_body . '.blade.php';
        }
        $companyProfile->quote_email_body            = $request->quote_email_body . '.blade.php';
        $companyProfile->quote_approved_email_body   = $request->quote_approved_email_body . '.blade.php';
        $companyProfile->quote_rejected_email_body   = $request->quote_rejected_email_body . '.blade.php';
        $companyProfile->invoice_email_body          = $request->invoice_email_body . '.blade.php';
        $companyProfile->credit_memos_email_body     = $request->credit_memos_email_body . '.blade.php';
        $companyProfile->overdue_invoice_email_body  = $request->overdue_invoice_email_body . '.blade.php';
        $companyProfile->payment_receipt_email_body  = $request->payment_receipt_email_body . '.blade.php';
        $companyProfile->upcoming_payment_email_body = $request->upcoming_payment_email_body . '.blade.php';
        if (config('sms'))
        {
            $companyProfile->quote_sms_body   = $request->quote_sms_body;
            $companyProfile->invoice_sms_body = $request->invoice_sms_body;
        }
        $companyProfile->save();

        // Save the custom fields.
        if ($companyProfile->custom)
        {
            $customFieldData = CustomFieldsTransformer::transform(($request->custom ?? []), 'company_profiles', $companyProfile);
            $companyProfile->custom->update($customFieldData);
        }
        else
        {
            $companyProfile->custom()->save(new CompanyProfileCustom());
            if (!empty($request->custom) && $companyProfile->custom)
            {
                $customFieldData = CustomFieldsTransformer::transform(($request->custom ?? []), 'company_profiles', $companyProfile);
                $companyProfile->custom->update($customFieldData);
            }
        }

        return redirect($this->getReturnUrl())
            ->with('alertSuccess', trans('fi.record_successfully_created'));
    }

    public function edit($id)
    {
        $companyProfile               = CompanyProfile::find($id);
        $emailList                    = Setting::emailList();
        $quoteEmailBody               = explode('.', $companyProfile->quote_email_body);
        $proformaInvoiceTemplateLists = (class_exists(\Addons\ProformaInvoice\Support\ProformaInvoiceTemplates::class));

        if (isset($companyProfile->proforma_invoice_email_body) && config('proforma_invoice_enabled') == true)
        {
            $proformaInvoiceEmailBody         = explode('.', $companyProfile->proforma_invoice_email_body);
            $proformaInvoiceApprovedEmailBody = explode('.', $companyProfile->proforma_invoice_approved_email_body);
            $proformaInvoiceRejectedEmailBody = explode('.', $companyProfile->proforma_invoice_rejected_email_body);
        }
        $quoteApprovedEmailBody   = explode('.', $companyProfile->quote_approved_email_body);
        $quoteRejectedEmailBody   = explode('.', $companyProfile->quote_rejected_email_body);
        $invoiceEmailBody         = explode('.', $companyProfile->invoice_email_body);
        $creditMemosEmailBody     = explode('.', $companyProfile->credit_memos_email_body);
        $overdueInvoiceEmailBody  = explode('.', $companyProfile->overdue_invoice_email_body);
        $paymentReceiptEmailBody  = explode('.', $companyProfile->payment_receipt_email_body);
        $upcomingPaymentEmailBody = explode('.', $companyProfile->upcoming_payment_email_body);


        return view('company_profiles.form')
            ->with('editMode', true)
            ->with('companyProfile', $companyProfile)
            ->with('proformaInvoiceTemplates', $proformaInvoiceTemplateLists == true ? \Addons\ProformaInvoice\Support\ProformaInvoiceTemplates::lists() : null)
            ->with('companyProfileInUse', CompanyProfile::inUse($id))
            ->with('invoiceTemplates', InvoiceTemplates::lists())
            ->with('quoteTemplates', QuoteTemplates::lists())
            ->with('proformaInvoiceTemplates', $proformaInvoiceTemplateLists == true ? \Addons\ProformaInvoice\Support\ProformaInvoiceTemplates::lists() : null)
            ->with('proformaInvoiceEmailType', $emailList['proformaInvoice']['MailTemplate'])
            ->with('proformaInvoiceApprovedEmailType', $emailList['proformaInvoiceApproved']['MailTemplate'])
            ->with('proformaInvoiceRejectedEmailType', $emailList['proformaInvoiceRejected']['MailTemplate'])
            ->with('quoteEmailType', $emailList['quote']['MailTemplate'])
            ->with('quoteApprovedEmailType', $emailList['quoteApproved']['MailTemplate'])
            ->with('quoteRejectedEmailType', $emailList['quoteRejected']['MailTemplate'])
            ->with('invoiceEmailType', $emailList['invoice']['MailTemplate'])
            ->with('creditMemosEmailType', $emailList['creditMemos']['MailTemplate'])
            ->with('overdueInvoiceEmailType', $emailList['overdueInvoice']['MailTemplate'])
            ->with('upcomingPaymentEmailType', $emailList['upcomingPayment']['MailTemplate'])
            ->with('paymentReceiptEmailType', $emailList['paymentReceipt']['MailTemplate'])
            ->with('selectedProformaInvoiceEmail', isset($proformaInvoiceEmailBody[0]) ? $proformaInvoiceEmailBody[0] : '')
            ->with('selectedProformaInvoiceApprovedEmail', isset($proformaInvoiceApprovedEmailBody[0]) ? $proformaInvoiceApprovedEmailBody[0] : '')
            ->with('selectedProformaInvoiceRejectedEmail', isset($proformaInvoiceRejectedEmailBody[0]) ? $proformaInvoiceRejectedEmailBody[0] : '')
            ->with('selectedQuoteEmail', $quoteEmailBody[0])
            ->with('selectedQuoteApprovedEmail', $quoteApprovedEmailBody[0])
            ->with('selectedQuoteRejectedEmail', $quoteRejectedEmailBody[0])
            ->with('selectedInvoiceEmail', $invoiceEmailBody[0])
            ->with('selectedUpcomingPaymentEmail', $upcomingPaymentEmailBody[0])
            ->with('selectedPaymentReceiptEmail', $paymentReceiptEmailBody[0])
            ->with('selectedCreditMemosEmail', $creditMemosEmailBody[0])
            ->with('selectedOverDueInvoiceEmail', $overdueInvoiceEmailBody[0])
            ->with('countries', Country::getAll())
            ->with('paymentTerms', PaymentTerm::getDropDownList())
            ->with('customFields', CustomFieldsParser::getFields('company_profiles'));

    }

    public function update(CompanyProfileUpdateRequest $request, $id)
    {

        $input   = $request->except('custom', 'quote_email_body', 'quote_approved_email_body', 'quote_rejected_email_body', 'invoice_email_body', 'credit_memos_email_body', 'overdue_invoice_email_body', 'payment_receipt_email_body', 'upcoming_payment_email_body');
        $manager = new ImageManager(new Driver());

        if ($request->hasFile('logo'))
        {
            $logoFileName = $request->file('logo')->getClientOriginalName();
            $request->file('logo')->move(company_profile_logo_path(), $logoFileName);

            if (file_exists(company_profile_logo_path() . DIRECTORY_SEPARATOR . $logoFileName))
            {

                ini_set('memory_limit', '540M');
                $manager->read(company_profile_logo_path() . DIRECTORY_SEPARATOR . $logoFileName)->resizeDown(1000, 1000)->save(company_profile_logo_path() . DIRECTORY_SEPARATOR . $logoFileName);
            }
            $input['logo'] = $logoFileName;
        }

        $companyProfile = CompanyProfile::find($id);
        $companyProfile->fill($input);
        $companyProfile->save();


        $companyProfile->quote_email_body = $request->quote_email_body . '.blade.php';

        if (isset($companyProfile->proforma_invoice_email_body) && config('proforma_invoice_enabled') == true)
        {
            $companyProfile->proforma_invoice_email_body          = $request->proforma_invoice_email_body . '.blade.php';
            $companyProfile->proforma_invoice_approved_email_body = $request->proforma_invoice_approved_email_body . '.blade.php';
            $companyProfile->proforma_invoice_rejected_email_body = $request->proforma_invoice_rejected_email_body . '.blade.php';
        }
        $companyProfile->quote_approved_email_body   = $request->quote_approved_email_body . '.blade.php';
        $companyProfile->quote_rejected_email_body   = $request->quote_rejected_email_body . '.blade.php';
        $companyProfile->invoice_email_body          = $request->invoice_email_body . '.blade.php';
        $companyProfile->credit_memos_email_body     = $request->credit_memos_email_body . '.blade.php';
        $companyProfile->overdue_invoice_email_body  = $request->overdue_invoice_email_body . '.blade.php';
        $companyProfile->payment_receipt_email_body  = $request->payment_receipt_email_body . '.blade.php';
        $companyProfile->upcoming_payment_email_body = $request->upcoming_payment_email_body . '.blade.php';
        if (config('sms'))
        {
            $companyProfile->quote_sms_body   = $request->quote_sms_body;
            $companyProfile->invoice_sms_body = $request->invoice_sms_body;
        }
        $companyProfile->save();

        // Save the custom fields.
        if ($companyProfile->custom)
        {
            $customFieldData = CustomFieldsTransformer::transform(($request->custom ?? []), 'company_profiles', $companyProfile);

            $companyProfile->custom->update($customFieldData);
        }
        else
        {
            $companyProfile->custom()->save(new CompanyProfileCustom());

            if (!empty($request->custom) && $companyProfile->custom)
            {
                $customFieldData = CustomFieldsTransformer::transform(($request->custom ?? []), 'company_profiles', $companyProfile);
                $companyProfile->custom->update($customFieldData);
            }
        }
        return redirect(route('company.profiles.index'))
            ->with('alertSuccess', trans('fi.record_successfully_updated'));
    }

    public function delete($id)
    {
        if (CompanyProfile::inUse($id))
        {
            return response()->json(['success' => false, 'message' => trans('fi.cannot_delete_record_in_use')], 400);
        }
        else
        {
            if (CompanyProfile::whereId($id)->whereIsDefault(0)->first())
            {
                CompanyProfile::destroy($id);
                return response()->json(['success' => true, 'message' => trans('fi.record_successfully_deleted')], 200);
            }
            else
            {
                return response()->json(['success' => false, 'message' => trans('fi.cannot_delete_default_company')], 400);
            }
        }
    }

    public function ajaxModalLookup()
    {
        return view('company_profiles._modal_lookup')
            ->with('id', request('id'))
            ->with('companyProfiles', CompanyProfile::getList())
            ->with('refreshFromRoute', request('refresh_from_route'))
            ->with('updateCompanyProfileRoute', request('update_company_profile_route'));
    }

    public function deleteLogo($id)
    {
        $companyProfile = CompanyProfile::find($id);

        $companyProfile->logo = null;

        $companyProfile->save();

        if (file_exists(company_profile_logo_path($companyProfile->logo)))
        {
            try
            {
                unlink(company_profile_logo_path($companyProfile->logo));
            }
            catch (Exception $e)
            {

            }
        }
    }

    public function deleteImage($id, $columnName)
    {
        $customFields = CompanyProfileCustom::whereCompanyProfileId($id)->first();

        $existingFile = 'company_profiles' . DIRECTORY_SEPARATOR . $customFields->{$columnName};
        if (Storage::disk(CustomFieldsTransformer::STORAGE_DISK_NAME)->exists($existingFile))
        {
            try
            {
                Storage::disk(CustomFieldsTransformer::STORAGE_DISK_NAME)->delete($existingFile);
                $customFields->{$columnName} = null;
                $customFields->save();
            }
            catch (Exception $e)
            {

            }
        }
    }

    public function deleteModal()
    {
        try
        {
            return view('layouts._delete_modal')->with('url', request('action'))->with('modalName', request('modalName'))->with('isReload', request('isReload'));
        }
        catch (\Exception $e)
        {
            return response()->json(['success' => false, 'message' => ($e->getCode() == 0) ? trans('fi.modal_not_found') : $e->getMessage()], 400);
        }
    }

    public function getPaymentTerm()
    {
        try
        {
            $companyProfile = CompanyProfile::find(request('company_profile_id'));
            $paymentTermId  = !empty($companyProfile->payment_term_id) ? $companyProfile->payment_term_id : config('fi.defaultPaymentTerms');
            return response()->json(['id' => $paymentTermId], 200);
        }
        catch (\Exception $e)
        {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 400);
        }
    }
}
