<?php

/**
 * This file is part of FusionInvoice.
 *
 * (c) SquarePig LLC <hello@squarepiginteractive.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

use FI\Modules\Clients\Models\Client;
use FI\Modules\Currencies\Models\Currency;
use FI\Modules\PaymentTerms\Models\PaymentTerm;
use FI\Modules\Settings\Models\UserSetting;
use FI\Support\CurrencyFormatter;
use FI\Support\DateFormatter;
use Illuminate\Support\Carbon;

function refactorColumnSetting($list_columns, $columns)
{
    foreach ($list_columns as $key => $index)
    {
        if (array_key_exists($index, request('columns')))
        {
            $columnsValues   = $columns[$index];
            $separatedValues = explode("###", $columnsValues);
            $columns[$index] = $separatedValues;
        }
        else
        {
            $columns[$index] = [0 => "off", 1 => "notSortable"];
        }
    }
    return json_encode($columns);
}

function refactorClientColumnSetting($column, $status)
{
    $userSettings = UserSetting::whereSettingKey($column)->whereUserId(auth()->user()->id)->first();
    if ($userSettings == null)
    {
        $userSetting                = new UserSetting();
        $userSetting->user_id       = auth()->user()->id;
        $userSetting->setting_key   = $column;
        $userSetting->setting_value = $status;
        $userSetting->save();
    }
    else
    {
        $userSettings->setting_value = $status;
        $userSettings->save();
    }
}

function onlinePaymentChargesDetail($invoice)
{
    $invoiceAmountBalance = floatval(number_format($invoice->amount->balance, 2, '.', ''));
    $feePercentage        = config('fi.feePercentage') != null ? (config('fi.feePercentage') / 100) : null;
    $feeCharges           = floatval(number_format($invoiceAmountBalance * $feePercentage, 2, '.', ''));
    $addChargeAsItem      = true;

    if ($invoice->online_payment_processing_fee == 'yes')
    {
        $amountValue = number_format(($invoiceAmountBalance + $feeCharges), 2, '.', '');
    }
    else
    {
        $addChargeAsItem = false;
        $amountValue     = $invoiceAmountBalance;
    }

    return paymentTermsDiscountDetail(
        [
            'percentage'           => 0,
            'type'                 => null,
            'originalAmountValue'  => $invoiceAmountBalance,
            'amountValue'          => floatval($amountValue),
            'addChargesItem'       => $addChargeAsItem,
            'feeCharges'           => $feeCharges,
            'paymentTermsDiscount' => 0.00,
            'installment'          => 0,
        ], $invoice);
}

function onlineConvenienceCharges($invoice, $feeCharges = null)
{
    if ($invoice->online_payment_processing_fee == 'yes' && config('fi.feePercentage') != '')
    {
        if ($feeCharges > 0 && $feeCharges != null)
        {
            $totalCharges                       = $invoice->total_convenience_charges + $feeCharges;
            $invoice->total_convenience_charges = number_format($totalCharges, 2, '.', '');
            $invoice->save();
        }
        else
        {
            $feePercentage                      = (config('fi.feePercentage') / 100);
            $feeCharges                         = (number_format($invoice->amount->balance, 2, '.', '') * $feePercentage);
            $totalCharges                       = $invoice->total_convenience_charges + $feeCharges;
            $invoice->total_convenience_charges = number_format($totalCharges, 2, '.', '');
            $invoice->save();
        }

    }
}

function manageTags($module, $tagUpdateAction, $tagDeleteAction, $model, $isDirtyNote = true)
{
    $addTags = $removeTags = null;

    $newTagsArray = request('tags', []);
    $oldTags      = $module->tags;
    $oldTagsArray = $oldNonIndexTagsArray = [];

    if (isset($oldTags) && isset($newTagsArray))
    {
        foreach ($oldTags as $oldTag)
        {
            $nameTag                       = $oldTag->tag->name;
            $oldNonIndexTagsArray[]        = $nameTag;
            $oldTagsArray[$oldTag->tag_id] = $nameTag;
        }

        if ($newTagsArray != $oldNonIndexTagsArray)
        {
            $addTags    = array_diff($newTagsArray, $oldTagsArray);
            $removeTags = array_diff($oldTagsArray, $newTagsArray);
        }
    }
    $eventPath = 'FI\Modules\\' . $model . '\Events\AddTransitionTags';
    if (isset($removeTags))
    {
        $removeTagIds = $removeTagNames = [];
        foreach ($removeTags as $removeTagId => $removeTag)
        {
            $removeTagIds[]   = $removeTagId;
            $removeTagNames[] = $removeTag;

        }
        if (!empty($removeTagIds))
        {
            event(new $eventPath($module, $tagDeleteAction, '', '', '', implode(', ', $removeTagNames), $removeTagIds, $isDirtyNote));
        }

    }
    if (isset($addTags))
    {
        $addTagNames = [];

        foreach ($addTags as $addTag)
        {
            $addTagNames[] = $addTag;
        }
        if (!empty($addTagNames))
        {
            event(new $eventPath($module, $tagUpdateAction, '', '', '', implode(', ', $addTagNames), '', $isDirtyNote));
        }
    }

    return $addTags;
}

function countColumns($defaultSequenceColumnData, $columnSettings)
{
    $columnIndex = 0;

    foreach ($defaultSequenceColumnData as $defaultColumnIndex => $defaultSequenceColumn)
    {
        foreach ($columnSettings as $columnSettingIndex => $columnSetting)
        {
            if ($defaultColumnIndex == $columnSettingIndex && $defaultSequenceColumn[0] == $columnSetting[0])
            {
                $columnIndex++;
                if ($defaultColumnIndex == 'total' && $defaultSequenceColumn[0] == 'on')
                {
                    return $columnIndex--;
                }
                else
                {
                    if ($defaultColumnIndex == 'balance' && $defaultSequenceColumn[0] == 'on')
                    {
                        return $columnIndex--;
                    }
                }
            }
        }
    }

    return 0;
}

function darkModeForInvoiceAndQuoteTemplate($htmlTemplate)
{
    return str_replace("table.table-dark tr:nth-child(odd) td {background: #2c3034;}  table.table-dark {color: #fff !important;background-color: #212529 !important;}  body.body-dark {color: #fff !important;background: #454d55 !important;}  body.body-dark a {color: #4ba1df;border-bottom: 1px solid currentColor;text-decoration: none;}  table.table-dark td, .table-dark th, .table-dark thead th {border-color: #32383e;color: white;}  table.body-dark .section-header {border-color: #32383e;color: white;}  table.body-dark .info {color: #b2b2b2;font-weight: bold;}  table.body-dark h1 {color: #b2b2b2;font-weight: bold;}", " ", $htmlTemplate);
}

function calculateDiscount($discount_type, $discount, $price)
{
    if ($discount_type == 'percentage')
    {
        return ['price' => $price - (($price * $discount) / 100), 'previous_price' => (($price * $discount) / 100)];
    }
    else if ($discount_type == 'flat_amount')
    {
        return ['price' => $price - $discount, 'previous_price' => $discount];
    }
    else
    {
        return ['price' => $price - $discount, 'previous_price' => 0];
    }
}

function removeCalculateDiscount($discount_type, $discount, $price)
{
    if ($discount_type == 'percentage')
    {
        return ['price' => $price + $discount];
    }
    else if ($discount_type == 'flat_amount')
    {
        return ['price' => $price + $discount];
    }
    else
    {
        return ['price' => $price];
    }
}

function userSettingUpdate($key, $value, $id)
{
    $userSettings = UserSetting::whereSettingKey($key)->whereUserId($id)->first();
    if ($userSettings == null)
    {
        $userSetting                = new UserSetting();
        $userSetting->user_id       = $id;
        $userSetting->setting_key   = $key;
        $userSetting->setting_value = $value;
        $userSetting->save();
    }
    else
    {
        $userSettings->setting_value = $value;
        $userSettings->save();
    }
}

function noteFormatter($id, $title, $detail)
{
    if (isset($detail->short_text))
    {
        $formatNote = str_replace(['&gt;', '&lt;', '&nbsp;'], ['', '', ''], strip_tags($detail->short_text));
        if (str_word_count(strip_tags($formatNote)) > 50)
        {
            return '<div id="module" class="note-container">' . $title . '
                        <div class="collapse note-collapse" id="collapse' . $id . '" aria-expanded="false">' . $detail->short_text . '</div>
                        <a role="button" class="collapsed note-collapsed" data-toggle="collapse" href="#collapse' . $id . '" aria-expanded="false" aria-controls="#collapse' . $id . '">' . trans("fi.show_more") . '</a>
                    </div>';
        }
        elseif (isset($detail->short_text))
        {
            return $title . '<br>' . trans('fi.text') . ': ' . $detail->short_text;
        }
        else
        {
            return $title;
        }
    }
}

function jsFormattedAddress($data)
{
    $patterns     = ["/\\\\/", '/\n/', '/\r/', '/\t/', '/\v/', '/\f/'];
    $replacements = ['\\\\\\', '\n', '\r', '\t', '\v', '\f'];
    $address      = preg_replace($patterns, $replacements, $data);
    return $address;
}

function iframeThemeColor()
{
    return '.theme-color {color: #000000 !important;}';
}

function revenueByClientCurrencyFormatter($amount = 0)
{
    return CurrencyFormatter::format($amount);
}

function getServiceStartDate($service_ends_of_the, $next, $service_starts_on)
{
    $next_date = Carbon::createFromDate($next);

    if ($service_ends_of_the == 'invoice_month_plus_one' || $service_ends_of_the == 'invoice_month_plus_two' || $service_ends_of_the == 'invoice_month_plus_three')
    {
        $number_array = ['one' => 1, 'two' => 2, 'three' => 3];
        $number       = $number_array[str_replace('invoice_month_plus_', '', $service_ends_of_the)];
        if ($service_starts_on == 1 && $service_starts_on > 0)
        {
            $start = $next_date->firstOfMonth()->addMonths($number)->firstOfMonth();
        }
        else
        {
            $start = $next_date->firstOfMonth()->addMonths($number)->addDays($service_starts_on)->subDay();
        }
    }
    elseif ($service_ends_of_the == 'next_year')
    {
        if ($service_starts_on == 1 && $service_starts_on > 0)
        {
            $start = $next_date->addYears()->firstOfYear();
        }
        else
        {
            $start = $next_date->addYears()->firstOfYear()->addDays($service_starts_on)->subDay();
        }
    }
    elseif ($service_ends_of_the == 'invoice_month' && $service_starts_on > 0)
    {
        if ($service_starts_on == 1)
        {
            $start = $next_date->firstOfMonth();
        }
        else
        {
            $start = $next_date->firstOfMonth()->addDays($service_starts_on)->subDay();

        }
    }
    elseif ($service_ends_of_the == 'invoice_year' && $service_starts_on > 0)
    {
        if ($service_starts_on == 1)
        {
            $start = $next_date->firstOfYear();
        }
        else
        {
            $start = $next_date->firstOfYear()->addDays($service_starts_on)->subDay();
        }
    }
    else
    {
        $start = $next_date->addDays($service_starts_on);
    }

    return $start->format(config('fi.dateFormat'));
}

function getServiceEndDate($start, $recurring_period, $recurring_frequency)
{
    $start_date = Carbon::createFromDate($start);
    $endDate = Carbon::createFromDate(DateFormatter::incrementDate(substr($start, 0, 10), $recurring_period, $recurring_frequency));

    return $endDate->format(config('fi.dateFormat'));
}

function paymentTermsDiscountDetail($details, $invoice, $addFeeCharge = true)
{
    $details['originalAmountValue'] = floatval(number_format($details['originalAmountValue'], 2, '.', ''));
    $details['amountValue']         = floatval(number_format($details['amountValue'], 2, '.', ''));
    $amountValue                    = $details['amountValue'];
    $currentDate                    = Carbon::now()->format('Y-m-d');
    $paymentTerm                    = $invoice->paymentTerms;
    $details['percentage']          = 0;
    $details['formattedZero']       = CurrencyFormatter::format(0.00, $invoice->currency);

    if ($invoice->amount != null && $paymentTerm != null)
    {
        $details['type'] = $paymentTerm->type;


        if ($currentDate < $invoice->formatted_payment_term_due_date['expire'])
        {
            if ($paymentTerm->type == 2 || $paymentTerm->type == 3)
            {
                $discount                        = floatval(number_format(((floatval($paymentTerm->discount_percent) * floatval($invoice->amount->balance)) / 100), 2, '.', ''));
                $details['paymentTermsDiscount'] = $discount;
                $details['percentage']           = floatval($paymentTerm->discount_percent);
                $details['installment']          = 0;
                if ($discount > 0)
                {
                    if ($details['addChargesItem'] == true)
                    {
                        $type2FeePercentage             = config('fi.feePercentage') != null ? (config('fi.feePercentage') / 100) : null;
                        $details['originalAmountValue'] = ($details['originalAmountValue'] - $discount);
                        $details['amountValue']         = floatval(number_format($details['originalAmountValue'] + ($details['originalAmountValue'] * $type2FeePercentage), 2, '.', ''));
                        $details['feeCharges']          = floatval(number_format(($details['originalAmountValue'] * $type2FeePercentage), 2, '.', ''));
                    }
                    else
                    {
                        $details['originalAmountValue'] = $details['amountValue'] = ($amountValue - $discount);
                    }
                }
            }
            elseif ($paymentTerm->type == 4)
            {
                $paymentTermDownPayment = floatval(number_format($paymentTerm->down_payment, 2, '.', ''));

                if ($paymentTerm->down_payment_type == 'percentage')
                {
                    $totalPayment = ($invoice->amount->total * ($paymentTermDownPayment / 100));

                    $paymentTermDownPayment = floatval(number_format($totalPayment, 2, '.', ''));
                }

                $totalPay = ($invoice->payments->count() > 0) ? floatval(number_format($invoice->payments()->sum('invoice_amount_paid'), 2, '.', '')) : 0;

                $flag = ($paymentTermDownPayment > $totalPay);

                if ($flag)
                {
                    $paymentTermDownPayment = floatval(number_format($paymentTermDownPayment - $totalPay, 2, '.', ''));
                }

                if (floatval($invoice->amount->balance) > $paymentTermDownPayment && $flag)
                {
                    $feePercentage                  = config('fi.feePercentage') != null ? (config('fi.feePercentage') / 100) : null;
                    $feeCharges                     = floatval(number_format(($paymentTermDownPayment * $feePercentage), 2, '.', ''));
                    $details['originalAmountValue'] = $paymentTermDownPayment;

                    if ($invoice->online_payment_processing_fee == 'yes' && $addFeeCharge == true)
                    {
                        $details['amountValue'] = floatval(number_format(($paymentTermDownPayment + $feeCharges), 2, '.', ''));
                        $details['feeCharges']  = $feeCharges;
                    }
                    else
                    {
                        $details['amountValue']    = $paymentTermDownPayment;
                        $details['addChargesItem'] = false;
                    }
                }

                $details['paymentTermsDiscount'] = 0;
                $details['installment']          = 0;
            }
            elseif ($paymentTerm->type == 5)
            {
                $installmentData                = $invoice->formatted_payment_term_due_date;
                $installmentPercent             = floatval($installmentData['installment_percent']);
                $totalPayBalance                = ($installmentData['lastPayment'] == true ? floatval(number_format(floatval($invoice->amount->balance), 2, '.', '')) : floatval(number_format(((floatval($invoice->amount->total) * $installmentPercent) / 100), 2, '.', '')));
                $details['originalAmountValue'] = $totalPayBalance;
                $feePercentage                  = config('fi.feePercentage') != null ? (config('fi.feePercentage') / 100) : null;
                $feeCharges                     = floatval(number_format($totalPayBalance, 2, '.', '') * $feePercentage);
                if ($invoice->online_payment_processing_fee == 'yes' && $addFeeCharge == true)
                {
                    $details['amountValue'] = floatval(number_format((number_format($totalPayBalance, 2, '.', '') + $feeCharges), 2, '.', ''));
                    $details['feeCharges']  = floatval(number_format($feeCharges, 2, '.', ''));
                }
                else
                {
                    $details['amountValue']    = floatval(number_format($totalPayBalance, 2, '.', ''));
                    $details['feeCharges']     = 0;
                    $details['addChargesItem'] = false;
                }
                $details['installment'] = $installmentData['id'];
            }
        }
    }
    else
    {
        $details['amountValue']         = floatval(number_format($details['amountValue'], 2, '.', ''));
        $details['originalAmountValue'] = floatval(number_format($details['originalAmountValue'], 2, '.', ''));
    }

    $details['formattedAmountValue']          = CurrencyFormatter::format($details['amountValue'], $invoice->currency);
    $details['formattedOriginalAmountValue']  = CurrencyFormatter::format($details['originalAmountValue'], $invoice->currency);
    $details['formattedPaymentTermsDiscount'] = CurrencyFormatter::format($details['paymentTermsDiscount'], $invoice->currency);
    $details['formattedFeeCharges']           = CurrencyFormatter::format($details['feeCharges'], $invoice->currency);

    return $details;

}

function getCurrentGitBranch($formatted = false)
{
    $gitBranchShow = (file_exists(base_path('git_status.php')));
    if ($gitBranchShow)
    {
        $gitStatusContent = file_get_contents(url('git_status.php'));
        $currentBranch    = explode('Current Branch:', $gitStatusContent);
        if (isset($currentBranch[1]) && $currentBranch[1] != null)
        {
            if ($formatted)
            {
                $getVersion = explode('/', strip_tags($currentBranch[1]));
                return (count($getVersion) >= 2 ? strip_tags($getVersion[1]) : strip_tags($currentBranch[1]));
            }
            return strip_tags($currentBranch[1]);
        }
    }
    return null;
}

function getReadMdFileContent()
{
    if (file_exists(base_path('TERMS.md')))
    {
        $markdownContent = file_get_contents(base_path('TERMS.md'));
        $parsedown       = new \Parsedown();
        return $parsedown->text($markdownContent);
    }
    else
    {
        return '';
    }
}

function currencyStringMatch($value)
{
    $matchScore             = 0;
    $allowedCurrencySymbols = Currency::getListBySymbol();
    $currencyCode           = 'USD';
    $currencySymbolsPattern = '[' . preg_quote(implode('', array_unique($allowedCurrencySymbols)), '/') . ']';
    $pattern                = '/(?:(?P<symbol>' . $currencySymbolsPattern . ')|(?P<code>' . $currencyCode . '))?\s*(?P<amount>[\d.,]+)/';
    $matches                = [];
    $isValid                = preg_match($pattern, $value, $matches);

    if ($isValid)
    {
        $cleanedString = preg_replace('/[0-9.,\s]/u', '', $value);

        if (strlen($cleanedString) == 0)
        {
            $matchScore = 1;

        }
        elseif (strlen($cleanedString) > 0)
        {
            $arrayResult = preg_split('//u', $cleanedString, -1, PREG_SPLIT_NO_EMPTY);

            if (in_array($arrayResult[0], $allowedCurrencySymbols))
            {
                $matchScore = 2;
            }
        }
    }

    return $matchScore;
}

function convertToDecimalFormat($value)
{

    $allowedCurrencySymbols = Currency::getListBySymbol();

    $currencySymbolsPattern = '[' . preg_quote(implode('', array_unique($allowedCurrencySymbols)), '/') . ']';

    $pattern = '/(?:(?P<symbol>' . $currencySymbolsPattern . ')|USD)?\s*(?P<amount>[\d.,]+)/';

    $matches = [];

    $isValid = preg_match($pattern, $value, $matches);

    if ($isValid)
    {
        $amount = $matches['amount'];

        $amount = str_replace(',', '.', $amount);

        if (substr_count($amount, ".") > 1)
        {
            $amount = preg_replace('/\.(?=.*\.)/', '', $amount);
        }

        // Extract the numeric part
        $numericValue = (float)number_format((float)preg_replace('/[^0-9.]/', '', $amount), 2, '.', '');

        return $numericValue;
    }

    return null;
}

function getCurrencyFetch($value)
{
    $cleanedString = preg_replace('/[0-9.,]/', '', $value);
    $arrayResult   = preg_split('//u', $cleanedString, -1, PREG_SPLIT_NO_EMPTY);
    return Currency::where('symbol', $arrayResult[0])->pluck('code')->toArray();
}

function getFreeSpace()
{
    $diskFreeSpace = disk_free_space(base_path());
    return round(($diskFreeSpace / 1000000), 2);
}

function demoLanguageAndTimezone()
{
    $languageAndTimezone = session()->get('demoLanguageAndTimezone');

    if (config('app.demo') == true && $languageAndTimezone != null)
    {
        return explode('###', $languageAndTimezone);
    }
    else
    {
        return null;
    }
}

function getClientName($id)
{
    $client = Client::find($id);
    return (isset($client->name)) ? $client->name : '';
}

function setForTestSmsRequiredKeys()
{

    return [
        'telnyx'     => (config('sms.telnyxApiKey') != '' && config('sms.telnyxFromNumber') != '' && config('sms.telnyxWebhookUrl') != ''),
        'click-send' => (config('sms.clickSendUsername') != '' && config('sms.clickSendApiKey') != ''),
        'twilio'     => (config('sms.twilioSid') != '' && config('sms.twilioToken') != '' && config('sms.twilioFromNumber') != '' && config('sms.twilioWebhookUrl') != '')
    ];

}

function yesNoArray()
{
    return ['0' => trans('fi.no'), '1' => trans('fi.yes')];
}

function invoicesDueAfter()
{
    $paymentTerm = PaymentTerm::where('is_default', 1)->first();
    return $paymentTerm->net_due_days ?? 0;
}

function findParentKeyByValue($array, $searchKey, $searchValue)
{
    $filteredArray = array_filter($array, function ($subArray) use ($searchKey, $searchValue)
    {
        return isset($subArray[$searchKey]) && $subArray[$searchKey] === $searchValue;
    });

    $parentKeys = array_keys($filteredArray);
    return count($parentKeys) > 0 ? $parentKeys[0] : null;
}

function findSupportTicketType($searchValue = null)
{
    $support_email_addresses = config('mailbox.support_email_addresses');
    $searchKey               = 'from_email';
    return findParentKeyByValue($support_email_addresses, $searchKey, $searchValue);
}